##
#### WORKPLACE PREPARATION ####
##


# Remove all previous objects
rm(list=ls(all=TRUE))

# Load packages for regression analysis 
# Prior to the first run, packages have to be installed using the "install.packages("PACKAGE_NAME")"comand
library(haven)
library(lme4)
library(sjstats)
library(stargazer)
library(pbkrtest)



##
#### IMPORT DATASET ####
##

# Import dataset "Export_Data.dta" (prepared in Stata) and define it as object "Dataset" in R
# Requires the package "haven"

Dataset <- read_dta("Export_Data.dta")


##
#### HYPOTHESIS 1 ####
# WORK EXPERIENCE AND GOAL CONFLICT #
##


# Model 1
# 'Fixed-Effects' = No pooling
#
# Model estimation
# Option -1  / No Intercept estimated, does not affect estimates
# Na.action = na.omit / Rows with one or more missing values are excluded from the analysis
m1 <- lm(formula=gc ~ ot + factor(id) -1 , 
         data=Dataset, 
         na.action=na.omit)

# Display estimation results
summary(m1)
# Display number of observations
nobs(m1)



# Store estimation results as exportable HTML table
# Requires the package "stargazer" for output formating
# For argument descriptions see https://cran.r-project.org/web/packages/stargazer/stargazer.pdf
# Accessed: 23. Sep 2019

outputm1 <- capture.output(stargazer(m1, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("factor"), 
                                     dep.var.labels   = "<b>Model 1</b><br><i>Fixed-Effects</i>",
                                     covariate.labels = "<b>Observational tenure</b>",
                                     add.lines = list(c("Study Waves", "T3,T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = ""
                                     ))


# Model 2
# 'Fixed-Effects' = No pooling
# With additional year-2014-indicator variable
#
# Model estimation
# For argument descriptions see model 1 annotation above
m2 <- lm(formula=gc ~ ot + year_14 + factor(id) -1 , 
         data=Dataset, 
         na.action=na.omit)

# Display estimation results
summary(m2)

# Display number of observations
nobs(m2)



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above

outputm2 <- capture.output(stargazer(m2, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("factor"), 
                                     dep.var.labels   = "<b>Model 2</b><br><i>Fixed-Effects</i>",
                                     covariate.labels = c("<b>Observational tenure</b>", "<b>Year 2014</b>"),
                                     add.lines = list(c("Study Waves", "T3,T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = ""
                                     ))




# Model 3
# Random intercept
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1


m3 <- lmer(formula=gc ~ ot + twe_04 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m3)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm3 <- p_value(m3, p.kr = TRUE) 
print(pm3)


# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m3 <- list(pm3[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm3 <- capture.output(stargazer(m3, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("factor","Constant"), 
                                     dep.var.labels   = "<b>Model 3</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>"),
                                     add.lines = list(c("Study Waves", "T3,T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m3
                                     ))





# Model 4
# Random intercept
# Includes year-2014-indicator variable as additional covariate
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m4 <- lmer(formula=gc ~ ot + twe_04 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)



# Display estimation results
summary(m4)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm4 <- p_value(m4, p.kr = TRUE) 
print(pm4)


# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m4 <- list(pm4[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm4 <- capture.output(stargazer(m4, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("factor","Constant"), 
                                     dep.var.labels   = "<b>Model 4</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>", "<b>Year 2014</b>"),
                                     add.lines = list(c("Study Waves", "T3,T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m4))





# Export Table C1.1 
# Showing results of models 1, 2, 3 and 4

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm1, between_models, outputm2, between_models, outputm3, between_models, outputm4, post_models, file="Tables/C1_1.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document




# Model 5
# Random intercept
# Includes personal characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m5 <- lmer(formula=gc ~ ot + twe_04 + age_04 + gender + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m5)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest" 

pm5 <- p_value(m5, p.kr = TRUE) 
print(pm5)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m5 <- list(pm5[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm5 <- capture.output(stargazer(m5, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant"), 
                                     dep.var.labels   = "<b>Model 5</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>", "<b>Age</b>","<b>Gender</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", ""), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m5))



# Model 6
# Random intercept
# Includes year-2014-indicator variable as additional covariate
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m6 <- lmer(formula=gc ~ ot + twe_04 + age_04 + gender + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m6)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm6 <- p_value(m6, p.kr = TRUE) 
print(pm6)


# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m6 <- list(pm6[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm6 <- capture.output(stargazer(m6, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant"), 
                                     dep.var.labels   = "<b>Model 6</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>", "<b>Age</b>","<b>Gender</b>", "<b>Year 2014</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", ""), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m6))






# Model 7
# Random intercept
# Includes professional characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m7 <- lmer(formula=gc ~ ot + twe_04 + age_04 + gender + stc + pt + tjc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)



# Display estimation results
summary(m7)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm7 <- p_value(m7, p.kr = TRUE) 
print(pm7)


# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m7 <- list(pm7[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm7 <- capture.output(stargazer(m7, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "stc", "tjc"), 
                                     dep.var.labels   = "<b>Model 7</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>", "<b>Age</b>","<b>Gender</b>", "<b>Part-time</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Specialist<br>Job changes"), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m7))



# Model 8
# Random intercept
# Includes year-2014-indicator variable as additional covariate
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m8 <- lmer(formula=gc ~ ot + twe_04 + age_04 + gender + stc + pt + tjc + year_14 + (1 | id), 
            data=Dataset, 
            na.action=na.omit)




# Display estimation results
summary(m8)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm8 <- p_value(m8, p.kr = TRUE) 
print(pm8)


# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m8 <- list(pm8[, 2])


# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm8 <- capture.output(stargazer(m8, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "stc", "tjc"),  
                                      dep.var.labels   = "<b>Model 8</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>", "<b>Age</b>","<b>Gender</b>", "<b>Part-time</b>","<b>Year 2014</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Specialist<br>Job changes"), c("Study waves", "T3, T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m8))



# Model 9
# Random intercept
# Includes leadership position as additional covariate
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m9 <- lmer(formula=gc ~ ot + twe_04 + age_04 + gender + stc + pt + tjc + lp + (1 | id), 
              data=Dataset, 
              na.action=na.omit)



# Display estimation results
summary(m9)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"


pm9 <- p_value(m9, p.kr = TRUE) 
print(pm9)



# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m9 <- list(pm9[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm9 <- capture.output(stargazer(m9, 
                                        type="html",
                                        style = "aer",
                                        keep.stat = "n",
                                        omit = c("Constant", "stc", "tjc", "lp"), 
                                        dep.var.labels   = "<b>Model 9</b><br><i>Random-Intercept</i>",
                                        covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>", "<b>Age</b>","<b>Gender</b>", "<b>Part-time</b>"),
                                        add.lines = list(c("Covariates without<br>significant estimates", "Specialist<br>Job changes<br>Leadership position"), c("Study waves", "T3, T4")),
                                        report = "vc*p",
                                        notes ="",
                                        notes.append = FALSE,
                                        notes.label = "",
                                        p=pvalues_m9))



# Model 10
# Random intercept
# Includes year-2014-indicator variable as additional covariate
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m10 <- lmer(formula=gc ~ ot + twe_04 + age_04 + gender + stc + pt + tjc + lp + year_14 + (1 | id), 
              data=Dataset, 
              na.action=na.omit)




# Display estimation results
summary(m10)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm10 <- p_value(m10, p.kr = TRUE) 
print(pm10)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m10 <- list(pm10[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm10 <- capture.output(stargazer(m10, 
                                        type="html",
                                        style = "aer",
                                        keep.stat = "n",
                                        omit = c("Constant", "stc", "tjc", "lp"), 
                                        dep.var.labels   = "<b>Model 10</b><br><i>Random-Intercept</i>",
                                        covariate.labels = c("<b>Observational tenure</b>", "<b>Pre-study tenure</b>", "<b>Age</b>","<b>Gender</b>", "<b>Part-time</b>", "<b>Year 2014</b>"),
                                        add.lines = list(c("Covariates without<br>significant estimates", "Specialist<br>Job changes<br>Leadership position"), c("Study waves", "T3, T4")),
                                        report = "vc*p",
                                        notes ="",
                                        notes.append = FALSE,
                                        notes.label = "",
                                        p=pvalues_m10))


# Export Table C1.2 
# Showing results of models 5, 6, 7, 8, 9 and 10

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm5, between_models, outputm6, between_models, outputm7, between_models, outputm8, between_models, outputm9, between_models, outputm10, post_models, file="Tables/C1_2.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document




##
#### HYPOTHESIS 2A ####
# GOAL CONFLICT AND JOB SATISFACTION #
##




# Model 1
# 'Fixed-Effects' = No pooling
#
# Model estimation
# Option -1  / No Intercept estimated, does not affect estimates
# Na.action = na.omit / Rows with one or more missing values are excluded from the analysis
m1 <- lm(formula=js ~ gc + year_05 + year_07 + year_14 + factor(id) -1 , 
         data=Dataset, 
         na.action=na.omit)

# Display estimation results
summary(m1)
# Display number of observations
nobs(m1)


# Store estimation results as exportable HTML table
# Requires the package "stargazer" for output formating
# For argument descriptions see https://cran.r-project.org/web/packages/stargazer/stargazer.pdf
# Accessed: 23. Sep 2019

outputm1 <- capture.output(stargazer(m1, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("year_05","year_07","factor"), 
                                     dep.var.labels   = "<b>Model 1</b><br><i>Fixed-Effects</i>",
                                     covariate.labels = c("<b>Goal conflict:Score</b>", "<b>Year 2014</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = ""))




# Model 2
# Random intercept
# Splits goal conflict scores into overall average and time-specific deviation from average
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m2 <- lmer(formula=js ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m2)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm2 <- p_value(m2, p.kr = TRUE) 
print(pm2)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m2 <- list(pm2[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above



outputm2 <- capture.output(stargazer(m2, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","year_05","year_07"), 
                                     dep.var.labels   = "<b>Model 2</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2014</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m2))



# Model 3
# Random intercept
# Includes personal characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m3 <- lmer(formula=js ~ dev_gc + avg_gc + age_04 + gender + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m3)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm3 <- p_value(m3, p.kr = TRUE) 
print(pm3)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m3 <- list(pm3[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm3 <- capture.output(stargazer(m3, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","year_05","year_07"), 
                                     dep.var.labels   = "<b>Model 3</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Age</b>", "<b>Gender</b>", "<b>Year 2014</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m3))



# Model 4
# Random intercept
# Includes professional characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m4 <- lmer(formula=js ~ dev_gc + avg_gc + pt + tjc + lp + year_05 + year_07 +  year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m4)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm4 <- p_value(m4, p.kr = TRUE) 
print(pm4)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m4 <- list(pm4[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm4 <- capture.output(stargazer(m4, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","pt","lp","year_05","year_07"), 
                                     dep.var.labels   = "<b>Model 4</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Job changes</b>","<b>Year 2014</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007<br>Part-time<br>Leadership position"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m4))



# Export Table C2a.1 
# Showing results of models 1, 2, 3 and 4

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm1, between_models, outputm2, between_models, outputm3, between_models, outputm4, post_models, file="Tables/C2a_1.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document




# Model 5
# Random intercept
# Includes work experience variables and specialist training completed variable as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m5 <- lmer(formula=js ~ dev_gc + avg_gc + pt + tjc + lp + stc + ot + twe_04 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m5)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"


pm5 <- p_value(m5, p.kr = TRUE) 
print(pm5)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m5 <- list(pm5[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm5 <- capture.output(stargazer(m5, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","stc","ot","twe_04","year_14"), 
                                     dep.var.labels   = "<b>Model 5</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Part-time</b>", "<b>Job changes</b>", "<b>Leadership position</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Specialist<br>Observational tenure<br>Pre-study tenure<br>Year 2014"), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m5))

# Model 6
# Random intercept
# Combines Model 3 and 4
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m6 <- lmer(formula=js ~ dev_gc + avg_gc + age_04 + gender + pt + tjc + lp + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m6)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm6 <- p_value(m6, p.kr = TRUE) 
print(pm6)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m6 <- list(pm6[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm6 <- capture.output(stargazer(m6, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","year_05","year_07","year_14"), 
                                     dep.var.labels   = "<b>Model 6</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Age</b>", "<b>Gender</b>", "<b>Part-time</b>", "<b>Job changes</b>", "<b>Leadership position</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m6))


# Model 7
# Random intercept
# Combines Model 3 and 5
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m7 <- lmer(formula=js ~ dev_gc + avg_gc + age_04 + gender + pt + tjc + lp  + stc + ot + twe_04 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m7)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm7 <- p_value(m7, p.kr = TRUE) 
print(pm7)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m7 <- list(pm7[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm7 <- capture.output(stargazer(m7, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","stc","ot","twe_04","year_14"), 
                                     dep.var.labels   = "<b>Model 7</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict: Deviation</b>","<b>Goal conflict: Average</b>", "<b>Age</b>", "<b>Gender</b>", "<b>Part-time</b>", "<b>Job changes</b>", "<b>Leadership position</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Specialist<br>Observational tenure<br>Pre-study tenure<br>Year 2014"), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m7))



# Export Table C2a.2 
# Showing results of models 5, 6 and 7

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm5, between_models, outputm6, between_models, outputm7, post_models, file="Tables/C2a_2.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document




# Model 8
# Random intercept
# Includes Age*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m8 <- lmer(formula=js ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + age_04 + age_04*dev_gc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m8)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm8 <- p_value(m8, p.kr = TRUE) 
print(pm8)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m8 <- list(pm8[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm8 <- capture.output(stargazer(m8, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_05", "year_07"), 
                                     dep.var.labels   = "<b>Model 8</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2014</b>","<b>Age</b>", "<b>(Age)*(Goal conflict deviation)</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m8))



# Model 9
# Random intercept
# Includes Gender*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m9 <- lmer(formula=js ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + gender + gender*dev_gc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m9)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm9 <- p_value(m9, p.kr = TRUE) 
print(pm9)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m9 <- list(pm9[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm9 <- capture.output(stargazer(m9, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_05", "year_07"), 
                                     dep.var.labels   = "<b>Model 9</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2014</b>", "<b>Gender</b>", "<b>(Gender)*(Goal conflict deviation)</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m9))



# Model 10
# Random intercept
# Includes Job changes*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m10 <- lmer(formula=js ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + tjc + tjc*dev_gc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m10)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm10 <- p_value(m10, p.kr = TRUE) 
print(pm10)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m10 <- list(pm10[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm10 <- capture.output(stargazer(m10, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_05", "year_07", "year_14"),  
                                     dep.var.labels   = "<b>Model 10</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Job changes</b>", "<b>(Job changes)*(Goal conflict deviation)</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m10))




# Model 11
# Random intercept
# Includes Leadership position*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m11 <- lmer(formula=js ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + lp + lp*dev_gc + (1 | id), 
            data=Dataset, 
            na.action=na.omit)


# Display estimation results
summary(m11)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm11 <- p_value(m11, p.kr = TRUE) 
print(pm11)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m11 <- list(pm11[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm11 <- capture.output(stargazer(m11, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "year_05", "year_07","year_14"), 
                                      dep.var.labels   = "<b>Model 11</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Leadership position</b>", "<b>(Leadership position)*(Goal conflict deviation)</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m11))




# Export Table C2a.3 
# Showing results of models 8, 9, 10 and 11

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm8, between_models, outputm9, between_models, outputm10, between_models, outputm11,  post_models, file="Tables/C2a_3.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document




##
#### HYPOTHESIS 2B ####
# GOAL CONFLICT AND WORK ENGAGEMENT #
##




# Model 1
# 'Fixed-Effects' = No pooling
#
# Model estimation
# Option -1  / No Intercept estimated, does not affect estimates
# Na.action = na.omit / Rows with one or more missing values are excluded from the analysis
m1 <- lm(formula=we ~ gc + year_05 + year_07 + year_14 + factor(id) -1 , 
         data=Dataset, 
         na.action=na.omit)

# Display estimation results
summary(m1)
# Display number of observations
nobs(m1)


# Store estimation results as exportable HTML table
# Requires the package "stargazer" for output formating
# For argument descriptions see https://cran.r-project.org/web/packages/stargazer/stargazer.pdf
# Accessed: 23. Sep 2019



outputm1 <- capture.output(stargazer(m1, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("factor", "year_05", "year_14"), 
                                     dep.var.labels   = "<b>Model 1</b><br><i>Fixed-Effects</i>",
                                     covariate.labels = c("<b>Goal Conflict</b>", "<b>Year 2007</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = ""))




# Model 2
# Random intercept
# Splits goal conflict scores into overall average and time-specific deviation from average
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m2 <- lmer(formula=we ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m2)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest" 

pm2 <- p_value(m2, p.kr = TRUE) 
print(pm2)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m2 <- list(pm2[, 2])


# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm2 <- capture.output(stargazer(m2, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_05", "year_14"),
                                     dep.var.labels   = "<b>Model 2</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Year 2007</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m2))



# Model 3
# Random intercept
# Includes personal characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m3 <- lmer(formula=we ~ dev_gc + avg_gc + age_04 + gender + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m3)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm3 <- p_value(m3, p.kr = TRUE) 
print(pm3)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m3 <- list(pm3[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm3 <- capture.output(stargazer(m3, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "age_04", "gender", "year_05", "year_14"), 
                                     dep.var.labels   = "<b>Model 3</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Year 2007</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Age<br>Gender<br>Year 2005<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m3))



# Model 4
# Random intercept
# Includes professional characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m4 <- lmer(formula=we ~ dev_gc + avg_gc + pt + tjc + lp + year_05 + year_07 +  year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m4)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm4 <- p_value(m4, p.kr = TRUE) 
print(pm4)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m4 <- list(pm4[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm4 <- capture.output(stargazer(m4, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "pt", "lp", "year_05", "year_14"), 
                                     dep.var.labels   = "<b>Model 4</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>","<b>Job changes</b>", "<b>Year 2007</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Part-time<br>Leadership position<br>Year 2005<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m4))



# Export Table C2b.1 
# Showing results of models 1, 2, 3 and 4

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm1, between_models, outputm2, between_models, outputm3, between_models, outputm4, post_models, file="Tables/C2b_1.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document





# Model 5
# Random intercept
# Includes work experience variables and specialist training completed variable as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m5 <- lmer(formula=we ~ dev_gc + avg_gc + pt + tjc + lp + stc + ot + twe_04 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m5)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm5 <- p_value(m5, p.kr = TRUE) 
print(pm5)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m5 <- list(pm5[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm5 <- capture.output(stargazer(m5, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "pt", "stc", "twe_04", "year_14"), 
                                     dep.var.labels   = "<b>Model 5</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Job changes</b>", "<b>Leadership position</b>", "<b>Observational tenure</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Part-time<br>Specialist<br>Pre-study tenure<br>Year 2014"), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m5))

# Model 6
# Random intercept
# Combines Model 3 and 4
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m6 <- lmer(formula=we ~ dev_gc + avg_gc + age_04 + gender + pt + tjc + lp + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m6)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm6 <- p_value(m6, p.kr = TRUE) 
print(pm6)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m6 <- list(pm6[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm6 <- capture.output(stargazer(m6, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "age_04", "gender", "pt", "year_05", "year_07", "year_14"), 
                                     dep.var.labels   = "<b>Model 6</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Job changes</b>", "<b>Leadership position</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Age<br>Gender<br>Part-time<br>Year 2005<br>Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m6))


# Model 7
# Random intercept
# Combines Model 3 and 5
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m7 <- lmer(formula=we ~ dev_gc + avg_gc + age_04 + gender + pt + tjc + lp  + stc + ot + twe_04 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m7)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm7 <- p_value(m7, p.kr = TRUE) 
print(pm7)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m7 <- list(pm7[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm7 <- capture.output(stargazer(m7, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "age_04", "gender", "pt", "stc", "twe_04", "year_14"), 
                                     dep.var.labels   = "<b>Model 7</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Job changes</b>", "<b>Leadership position</b>", "<b>Observational tenure</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Age<br>Gender<br>Part-time<br>Specialist<br>Pre-study tenure<br>Year 2014"), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m7))



# Export Table C2b.2 
# Showing results of models 5, 6 and 7

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm5, between_models, outputm6, between_models, outputm7, post_models, file="Tables/C2b_2.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document





# Model 8
# Random intercept
# Includes Age*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m8 <- lmer(formula=we ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + age_04 + age_04*dev_gc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m8)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm8 <- p_value(m8, p.kr = TRUE) 
print(pm8)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m8 <- list(pm8[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm8 <- capture.output(stargazer(m8, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_05", "year_14"), 
                                     dep.var.labels   = "<b>Model 8</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2007</b>","<b>Age</b>", "<b>(Age)*(Goal conflict deviation)</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m8))



# Model 9
# Random intercept
# Includes Gender*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m9 <- lmer(formula=we ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + gender + gender*dev_gc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m9)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm9 <- p_value(m9, p.kr = TRUE) 
print(pm9)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m9 <- list(pm9[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above

outputm9 <- capture.output(stargazer(m9, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_05", "year_14"), 
                                     dep.var.labels   = "<b>Model 9</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2007</b>", "<b>Gender</b>", "<b>(Gender)*(Goal conflict deviation)</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m9))



# Model 10
# Random intercept
# Includes Job changes*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m10 <- lmer(formula=we ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + tjc + tjc*dev_gc + (1 | id), 
            data=Dataset, 
            na.action=na.omit)


# Display estimation results
summary(m10)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm10 <- p_value(m10, p.kr = TRUE) 
print(pm10)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m10 <- list(pm10[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm10 <- capture.output(stargazer(m10, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "year_05", "year_07","year_14"), 
                                      dep.var.labels   = "<b>Model 10</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Job changes</b>", "<b>(Job changes)*(Goal conflict deviation)</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m10))




# Model 11
# Random intercept
# Includes Leadership position*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m11 <- lmer(formula=we ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + lp + lp*dev_gc + (1 | id), 
            data=Dataset, 
            na.action=na.omit)


# Display estimation results
summary(m11)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm11 <- p_value(m11, p.kr = TRUE) 
print(pm11)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m11 <- list(pm11[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm11 <- capture.output(stargazer(m11, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "year_05", "year_14"), 
                                      dep.var.labels   = "<b>Model 11</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2007</b>", "<b>Leadership position</b>", "<b>(Leadership position)*(Goal conflict deviation)</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Year 2005<br>Year 2014"), c("Study waves", "T1 - T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m11))




# Export Table C2b.3 
# Showing results of models 8, 9, 10 and 11

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm8, between_models, outputm9, between_models, outputm10, between_models, outputm11,  post_models, file="Tables/C2b_3.html")


# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document





##
#### HYPOTHESIS 2C ####
# GOAL CONFLICT AND DEPRESSIVE SYMPTOMS 
##



# Model 1
# 'Fixed-Effects' = No pooling
#
# Model estimation
# Option -1  / No Intercept estimated, does not affect estimates
# Na.action = na.omit / Rows with one or more missing values are excluded from the analysis
m1 <- lm(formula=de ~ gc + year_05 + year_07 + year_14 + factor(id) -1 , 
         data=Dataset, 
         na.action=na.omit)

# Display estimation results
summary(m1)
# Display number of observations
nobs(m1)


# Store estimation results as exportable HTML table
# Requires the package "stargazer" for output formating
# For argument descriptions see https://cran.r-project.org/web/packages/stargazer/stargazer.pdf
# Accessed: 23. Sep 2019

outputm1 <- capture.output(stargazer(m1, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("factor", "year_07", "year_14"),
                                     dep.var.labels   = "<b>Model 1</b><br><i>Fixed-Effects</i>",
                                     covariate.labels = c("<b>Goal Conflict</b>", "<b>Year 2005</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = ""))




# Model 2
# Random intercept
# Splits goal conflict scores into overall average and time-specific deviation from average
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m2 <- lmer(formula=de ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m2)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest" 

pm2 <- p_value(m2, p.kr = TRUE) 
print(pm2)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m2 <- list(pm2[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm2 <- capture.output(stargazer(m2, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_07", "year_14"),
                                     dep.var.labels   = "<b>Model 2</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Year 2005</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m2))



# Model 3
# Random intercept
# Includes personal characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m3 <- lmer(formula=de ~ dev_gc + avg_gc + age_04 + gender + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m3)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm3 <- p_value(m3, p.kr = TRUE) 
print(pm3)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m3 <- list(pm3[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm3 <- capture.output(stargazer(m3, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant", "year_07", "year_14"),
                                     dep.var.labels   = "<b>Model 3</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Age</b>", "<b>Gender</b>", "<b>Year 2005</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m3))



# Model 4
# Random intercept
# Includes professional characteristics as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m4 <- lmer(formula=de ~ dev_gc + avg_gc + pt + tjc + lp + year_05 + year_07 +  year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m4)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm4 <- p_value(m4, p.kr = TRUE) 
print(pm4)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m4 <- list(pm4[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm4 <- capture.output(stargazer(m4, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","lp", "year_07", "year_14"),
                                     dep.var.labels   = "<b>Model 4</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Part-time</b>", "<b>Job changes</b>","<b>Year 2005</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Leadership position<br>Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m4))



# Export Table C2c.1 
# Showing results of models 1, 2, 3 and 4

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm1, between_models, outputm2, between_models, outputm3, between_models, outputm4, post_models, file="Tables/C2c_1.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document




# Model 5
# Random intercept
# Includes work experience variables and specialist training completed variable as additional covariates
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m5 <- lmer(formula=de ~ dev_gc + avg_gc + pt + tjc + lp + stc + ot + twe_04 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m5)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm5 <- p_value(m5, p.kr = TRUE) 
print(pm5)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m5 <- list(pm5[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm5 <- capture.output(stargazer(m5, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","pt", "stc", "ot", "twe_04", "year_14"), 
                                     dep.var.labels   = "<b>Model 5</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Job changes</b>", "<b>Leadership position</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Part-time<br>Specialist<br>Observational tenure<br>Pre-study tenure<br>Year 2014"), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m5))

# Model 6
# Random intercept
# Combines Model 3 and 4
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m6 <- lmer(formula=de ~ dev_gc + avg_gc + age_04 + gender + pt + tjc + lp + year_05 + year_07 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m6)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm6 <- p_value(m6, p.kr = TRUE) 
print(pm6)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m6 <- list(pm6[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm6 <- capture.output(stargazer(m6, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","age_04", "pt", "year_07", "year_14"), 
                                     dep.var.labels   = "<b>Model 6</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Gender</b>", "<b>Job changes</b>", "<b>Leadership position</b>","<b>Year 2005</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Age<br>Part-time<br>Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m6))


# Model 7
# Random intercept
# Combines Model 3 and 5
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m7 <- lmer(formula=de ~ dev_gc + avg_gc + age_04 + gender + pt + tjc + lp  + stc + ot + twe_04 + year_14 + (1 | id), 
           data=Dataset, 
           na.action=na.omit)




# Display estimation results
summary(m7)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest" 

pm7 <- p_value(m7, p.kr = TRUE) 
print(pm7)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m7 <- list(pm7[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm7 <- capture.output(stargazer(m7, 
                                     type="html",
                                     style = "aer",
                                     keep.stat = "n",
                                     omit = c("Constant","age_04", "pt", "stc", "ot", "twe_04", "year_14"), 
                                     dep.var.labels   = "<b>Model 7</b><br><i>Random-Intercept</i>",
                                     covariate.labels = c("<b>Goal Conflict: Deviation</b>","<b>Goal Conflict: Average</b>", "<b>Gender</b>", "<b>Job changes</b>", "<b>Leadership position</b>"),
                                     add.lines = list(c("Covariates without<br>significant estimates", "Age<br>Part-time<br>Specialist<br>Observational tenure<br>Pre-study tenure<br>Year 2014"), c("Study waves", "T3, T4")),
                                     report = "vc*p",
                                     notes ="",
                                     notes.append = FALSE,
                                     notes.label = "",
                                     p=pvalues_m7))



# Export Table C2c.2 
# Showing results of models 5, 6 and 7

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm5, between_models, outputm6, between_models, outputm7, post_models, file="Tables/C2c_2.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document




# Model 8
# Random intercept
# Includes Age*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m8 <- lmer(formula=de ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + age_04 + age_04*dev_gc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m8)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm8 <- p_value(m8, p.kr = TRUE) 
print(pm8)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m8 <- list(pm8[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above

outputm8 <- capture.output(stargazer(m8, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "year_07", "year_14"),  
                                      dep.var.labels   = "<b>Model 8</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2005</b>", "<b>Age</b>", "<b>(Age)*(Goal conflict deviation)</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m8))




# Model 9
# Random intercept
# Includes Gender*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m9 <- lmer(formula=de ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + gender + gender*dev_gc + (1 | id), 
           data=Dataset, 
           na.action=na.omit)


# Display estimation results
summary(m9)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm9 <- p_value(m9, p.kr = TRUE) 
print(pm9)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m9 <- list(pm9[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm9 <- capture.output(stargazer(m9, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "year_07", "year_14"), 
                                      dep.var.labels   = "<b>Model 9</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2005</b>", "<b>Gender</b>", "<b>(Gender)*(Goal conflict deviation)</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m9))



# Model 10
# Random intercept
# Includes Job changes*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m10 <- lmer(formula=de ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + tjc + tjc*dev_gc + (1 | id), 
            data=Dataset, 
            na.action=na.omit)


# Display estimation results
summary(m10)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm10 <- p_value(m10, p.kr = TRUE) 
print(pm10)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m10 <- list(pm10[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm10 <- capture.output(stargazer(m10, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "year_07", "year_14"),  
                                      dep.var.labels   = "<b>Model 10</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2005</b>", "<b>Job changes</b>", "<b>(Job changes)*(Goal conflict deviation)</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Year 2007<br>Year 2014"), c("Study waves", "T1 - T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m10))



# Model 11
# Random intercept
# Includes Leadership position*Goal Conflict Deviation interaction
#
# Model estimation
# Requires the package "lme4"  for model estimation
# Other arguments, see model 1

m11 <- lmer(formula=de ~ dev_gc + avg_gc + year_05 + year_07 + year_14 + lp + lp*dev_gc + (1 | id), 
            data=Dataset, 
            na.action=na.omit)


# Display estimation results
summary(m11)

# Compute p-values  
#
# Requires the package "sjstats"  for test statistics
# p.kr='TRUE'  / Using the Kenward Roger method to compute degrees of freedom 
# Requires installation of the package "pbkrtest"

pm11 <- p_value(m11, p.kr = TRUE) 
print(pm11)

# Integrate Kennward-Roger based p-values into output table
# Convert p-values dataframe into list format (required input type for ouput table creation, see below)
pvalues_m11 <- list(pm11[, 2])



# Store estimation results as exportable HTML table
# For argument descriptions see model 1 annotation above


outputm11 <- capture.output(stargazer(m11, 
                                      type="html",
                                      style = "aer",
                                      keep.stat = "n",
                                      omit = c("Constant", "year_07"), 
                                      dep.var.labels   = "<b>Model 11</b><br><i>Random-Intercept</i>",
                                      covariate.labels = c("<b>Goal conflict:Deviation</b>","<b>Goal conflict:Average</b>", "<b>Year 2005</b>", "<b>Year 2014</b>", "<b>Leadership position</b>", "<b>(Leadership position)*(Goal conflict deviation)</b>"),
                                      add.lines = list(c("Covariates without<br>significant estimates", "Year 2007"), c("Study waves", "T1 - T4")),
                                      report = "vc*p",
                                      notes ="",
                                      notes.append = FALSE,
                                      notes.label = "",
                                      p=pvalues_m11))





# Export Table C2c.3 
# Showing results of models 8, 9, 10 and 11

# Define HTML linking elements
pre_models = "<table><tr><td>"
between_models ="</td><td>"
post_models = "</td></tr></table>"


# Compile table and export as HTML file
cat(pre_models, outputm8, between_models, outputm9, between_models, outputm10, between_models, outputm11,  post_models, file="Tables/C2c_3.html")

# Subsequent steps:
# HTML file is imported into Microsoft Excel 
# Formatting is improved (without altering data) and table is saved as Excel file
# Excel file is integrated into main text document



