package eu.ascens.helenaText.util

import java.util.List
import org.eclipse.xtext.common.types.JvmType
import org.eclipse.xtext.common.types.TypesFactory

class JvmTypesHelper {
	
	/**
	 * Creats a JvmGenericType with simpleName set to className
	 */
	def static createJvmType(String className) {
		val declaredType = TypesFactory::eINSTANCE.createJvmGenericType
		declaredType.setSimpleName(className)
		declaredType
	}
	
	/**
	 * Return the wrapped JvmType (e.g. Integer for int)
	 */
	public def static JvmType getWrappedType(JvmType type){
		var wrapperName = type.simpleName.wrappedTypeAsString;
		createJvmType(wrapperName);
	}
	
	
	/**
	 * Return the wrapper classname for a primitive datatype (like e.g. int);
	 * e.g. if dataType is "int" -> the function returns Integer
	 */
	public def static String getWrappedTypeAsString(String typeName) {
		switch typeName {
			case "byte": return "Byte"
			case "short": return "Short"
			case "int": return "Integer"
			case "double" : return "Double"
			case "boolean": return "Boolean"
			case "char": return "Character"
			case "void": return "Void"
			//checks for arrays: wrapps first type part of the array, appends array brackets afterwards
			case typeName.contains("["): return getWrappedTypeAsString(typeName.substring(0, typeName.indexOf("["))) + typeName.substring(typeName.indexOf("["))
			default: return typeName
		}
	}
	
	public def static String getUnwrapperAsString(JvmType type) {
		switch type.simpleName {
			case "byte" : return ".byteValue()"
			case "Byte" : return ".byteValue()"
			case "short": return ".shortValue()"
			case "Short": return ".shortValue()"
			case "int": return ".intValue()"
			case "Integer": return ".intValue()"
			case "double" : return ".doubleValue()"
			case "Double" : return ".doubleValue()"
			case "boolean": return ".booleanValue()"
			case "Boolean": return ".booleanValue()"
			case "char": return ".charValue()"
			case "Character": return ".charValue()"
			default: return ""
		}
	}
	
	/**
	 * Compares two JvmTypes
	 * @returns true if both JvmTypes do have the same simpleName <br>
	 * false otherwise
	 */
	def static boolean typesEqual(JvmType first, JvmType second){
		return first.simpleName == second.simpleName;
	}
	/**
	 * Compares two Lists of JvmTypes
	 * @returns true if the elements from both lists do have the same simpleName <br>
	 * false otherwise
	 */
	def static boolean typesEqual(List<JvmType> list1, List<JvmType> list2) {
		if ( (list1 == null && list2 == null) ||
			(list1.size == 0 && list2.size == 0) ) {
			return true;
		}
		
		if (list1.size != list2.size) {
			return false;
		} 
		
		for (i : 0 ..< list1.size) {
			var first = list1.get(i);
			var second = list2.get(i);
			if (first.simpleName != second.simpleName) {
				return false;
			}
		}
		return true;
	}
}