package eu.ascens.helena.metadata;

import eu.ascens.helena.dev.Component;
import eu.ascens.helena.dev.exceptions.ConfigurationFinishedException;
import eu.ascens.helena.dev.exceptions.TypeAlreadyExistsException;
import eu.ascens.helena.dev.exceptions.TypeDoesNotExistException;

/**
 * This class represents component association types in Helena
 * 
 * @author Annabelle Klarl
 *
 */
public class ComponentAssociationType extends AbstractFieldType<Component> {

	/**
	 * This method creates a component association type for the given name and
	 * type.
	 * 
	 * If there is none for the name and type, a new one is created. If there
	 * exists already one, an {@link TypeAlreadyExistsException} is thrown.
	 * 
	 * @param name
	 *            the name of the component association
	 * @param type
	 *            the type of the component association
	 * @return
	 * @throws ConfigurationFinishedException This exception is thrown if this
	 *             method was called after the configuration of the
	 *             ensemble-based system was finished.
	 * @throws TypeAlreadyExistsException This exception is thrown if a type
	 *             with the same name already exists.
	 */
	public static ComponentAssociationType createType(String name,
	        Class<? extends Component> type)
	                throws ConfigurationFinishedException,
	                TypeAlreadyExistsException {

		ComponentAssociationType newAssociationType = new ComponentAssociationType(
		        name, type);
		AbstractFieldType.checkAndAddType(newAssociationType);
		return newAssociationType;
	}

	/**
	 * This method retrieves a component association type for the given name.
	 * 
	 * 
	 * @return
	 * @throws TypeDoesNotExistException This exception is thrown if there
	 *             exists not type for the given class.
	 */
	public static ComponentAssociationType getType(String name)
	        throws TypeDoesNotExistException {
		return AbstractFieldType.getType(name, ComponentAssociationType.class);
	}

	// //////// Component Association Type Declaration //////////////////////

	private ComponentAssociationType(String name,
	        Class<? extends Component> type) {
		super(name, type);
	}
}
