package eu.ascens.helena.dev;

import eu.ascens.helena.dev.exceptions.PropertyNotDeclaredInClassException;
import eu.ascens.helena.dev.exceptions.ReflectionException;
import eu.ascens.helena.metadata.ComponentType;
import eu.ascens.helena.metadata.OperationType;

/**
 * This class represents the action of calling an operation on a component in
 * Helena.
 *
 * @author Annabelle Klarl
 */
public class OperationCallAction<T> extends Action {

	private final Variable<T> returnValue;
	private final Operation op;
	private final Class<T> returnType;

	public OperationCallAction(Variable<T> returnValue, Operation op,
			Class<T> returnType) {
		this.returnValue = returnValue;
		this.op = op;
		this.returnType = returnType;
	}

	/**
	 * This methods calls the given operation on the owning component of the
	 * source role. The actual values for the parameters are given in the
	 * operation. It set the return value in the given return variable of this
	 * class.
	 *
	 * @param source the role issuing the action
	 * @throws PropertyNotDeclaredInClassException This exception is thrown when
	 *             the operation is not declared in the owning component
	 * @throws ReflectionException This exception is thrown if the operation
	 *             could not be called via reflection.
	 */
	@Override
	void execute(Role source)
			throws PropertyNotDeclaredInClassException, ReflectionException {

		OperationType opType = this.op.getType();
		ComponentType compType = source.getOwner().getType();

		if (!compType.isAllowed(opType)
				|| (opType.getReturnType() != this.returnType)) {
			throw new PropertyNotDeclaredInClassException(
					opType.getSimpleName(), compType.getType());

		}

		T value = source.getOwner().callOperation(this.op, this.returnType);
		this.returnValue.setValue(value);

		this.log.info(
				"Role " + source.getType() + " called operation " + opType);
	}

}
