package eu.ascens.helena.metadata;

import java.util.HashMap;
import java.util.Map;

import eu.ascens.helena.dev.Ensemble;
import eu.ascens.helena.dev.exceptions.ConfigurationFinishedException;
import eu.ascens.helena.dev.exceptions.TypeAlreadyExistsException;

/**
 * This class represents ensemble structures in Helena.
 *
 * @author Annabelle Klarl
 */
public class EnsembleStructure extends AbstractHelenaType<Ensemble> {

	private static Map<Class<? extends Ensemble>, EnsembleStructure> types = new HashMap<>();

	/**
	 * This method creates an ensemble structure for the given implementing
	 * class and the associated role types (together with their multiplicities
	 * and capacities).
	 * 
	 * If there is none for the implementing class, a new one is created. If
	 * there exists already one, an {@link TypeAlreadyExistsException} is
	 * thrown.
	 *
	 * @param name the name of the implementing class
	 * @param roleTypes the role types that are allowed in the ensemble
	 *            structure together with their associated multiplicities and
	 *            capacities
	 * @return
	 * @throws ConfigurationFinishedException This exception is thrown if this
	 *             method was called after the configuration of the
	 *             ensemble-based system was finished.
	 * @throws TypeAlreadyExistsException This exception is thrown if a type
	 *             with the same name already exists.
	 */
	public static EnsembleStructure createType(Class<? extends Ensemble> name,
	        Map<RoleType, MultiplicityCapacityPair> roleTypes) {
		EnsembleStructure es = new EnsembleStructure(name, roleTypes);
		AbstractHelenaType.checkAndAddType(es, types);
		return es;
	}

	/**
	 * This method retrieves an ensemble structure for the given implementing
	 * class.
	 *
	 * @param name the name of the implementing class
	 * @return
	 * @throws TypeAlreadyExistsException
	 */
	public static EnsembleStructure getType(Class<? extends Ensemble> name)
	        throws TypeAlreadyExistsException {
		return AbstractHelenaType.getType(name, types);
	}

	// //////////// Ensemble Structure Declaration ///////////////////////

	private final Map<RoleType, MultiplicityCapacityPair> roleTypes;

	private EnsembleStructure(Class<? extends Ensemble> name,
	        Map<RoleType, MultiplicityCapacityPair> roleTypes) {
		super(name);
		this.roleTypes = roleTypes;
	}

	/**
	 * This method checks whether an instance of the ensemble can host instance
	 * of the given role type.
	 *
	 * @param roleType
	 * @return
	 */
	public boolean isAllowed(RoleType roleType) {
		return this.roleTypes.containsKey(roleType);
	}

	/**
	 * This method returns the maximum number of role instances that are allowed
	 * for the given role type.
	 *
	 * @param roleType
	 * @return
	 */
	public int getMultiplicity(RoleType roleType) {
		MultiplicityCapacityPair pair = this.roleTypes.get(roleType);
		if (pair == null) {
			return 0;
		}
		else {
			return pair.getMultiplicity();
		}
	}

	/**
	 * This method returns the capacity for the given role type.
	 *
	 * @param roleType
	 * @return
	 */
	public int getCapacity(RoleType roleType) {
		MultiplicityCapacityPair pair = this.roleTypes.get(roleType);
		if (pair == null) {
			return 0;
		}
		else {
			return pair.getCapacity();
		}
	}
}
