package eu.ascens.helena.dev;

import eu.ascens.helena.dev.exceptions.ActionCurrentlyNotExecutableException;
import eu.ascens.helena.dev.exceptions.ActionNeverExecutableException;
import eu.ascens.helena.dev.exceptions.GuardNeverEvaluableException;
import eu.ascens.helena.dev.exceptions.InvokedProcessNotSetException;
import eu.ascens.helena.dev.exceptions.WellFormednessViolatedException;

/**
 * This is the abstract super class for all process expressions in Helena.
 *
 * @author Annabelle Klarl
 */
public abstract class ProcessExpression {

	/**
	 * This method performs the next executable action of the process expression
	 * for the given role. It returns the remaining process expression to be
	 * executed after executing the action.
	 *
	 * @param source the role executing the process expression
	 * @return the process expression remaining to be executed after this action
	 * @throws WellFormednessViolatedException This exception is thrown if the
	 *             action to be executed was not well-formed according to the
	 *             well-formedness conditions of Helena (e.g., message was not
	 *             allowed as output or input, message exchange between roles of
	 *             two roles in different ensembles).
	 * @throws ActionCurrentlyNotExecutableException This exception is thrown if
	 *             there is no action which could currently not be executed, but
	 *             at a later point in time the next action might become
	 *             executable (e.g., a message should have been received, but
	 *             the input channel was empty or a message should have been
	 *             sent, but the input channel was full).
	 * @throws ActionNeverExecutableException This exception is thrown if there
	 *             is no action which could be executed and the next action will
	 *             never become executable (e.g., a message should have been
	 *             sent or received on an input channel which was already
	 *             closed, a message was received which was not expected).
	 * @throws GuardNeverEvaluableException This exception is thrown if the
	 *             guard of if-then-else could not be evaluated since either it
	 *             referred to attributes of roles/components which cannot be
	 *             accessed or the guard did not evaluate to true.
	 * @throws InvokedProcessNotSetException This exception is thrown if a
	 *             process invocation was used, but the invoked process was not
	 *             set.
	 */
	abstract ProcessExpression step(Role source)
	        throws WellFormednessViolatedException,
	        ActionCurrentlyNotExecutableException,
	        ActionNeverExecutableException, GuardNeverEvaluableException,
	        InvokedProcessNotSetException;

}
