package eu.ascens.helena.dev;

import java.util.Set;

import eu.ascens.helena.dev.exceptions.NoBooleanValueException;
import eu.ascens.helena.dev.exceptions.PropertyNotDeclaredInClassException;
import eu.ascens.helena.dev.exceptions.ReflectionException;

/**
 * This class represents the disjunction of two guards.
 *
 * @author Annabelle Klarl
 */
public class OrGuard extends Guard {

	private final Guard left;
	private final Guard right;

	public OrGuard(Guard left, Guard right) {
		this.left = left;
		this.right = right;
	}

	@Override
	Set<Component> lockObjects(Role source) {
		return Auxiliaries.unify(this.left.lockObjects(source),
				this.right.lockObjects(source));
	}

	/**
	 * This method evaluates whether the disjunction of two guards is true.
	 *
	 * @param source the role for which the guard should be evaluated
	 * @return
	 * @throws PropertyNotDeclaredInClassException This exception is thrown if
	 *             the guard evaluates a property which is not defined in its
	 *             base class.
	 * @throws ReflectionException This exception is thrown if the guard
	 *             evaluates a property which could not be retrieved via
	 *             reflection.
	 * @throws NoBooleanValueException This exception is thrown if the guard did
	 *             not return a boolean value.
	 */
	@Override
	boolean isTrue(Role source) throws PropertyNotDeclaredInClassException,
			ReflectionException, NoBooleanValueException {
		return this.left.isTrue(source) || this.right.isTrue(source);
	}
}
