package eu.ascens.helena.dev;

import java.util.Set;
import java.util.logging.Logger;

import eu.ascens.helena.dev.exceptions.PropertyNotDeclaredInClassException;
import eu.ascens.helena.dev.exceptions.ReflectionException;
import eu.ascens.helena.metadata.ComponentType;
import eu.ascens.helena.metadata.DataFieldType;

/**
 * This class represents a getter for a component attribute of the owner of a
 * source role.
 *
 * @author Annabelle Klarl
 */
public class CompAttrGetter<T> extends Expression {

	protected final Logger log = Logger
			.getLogger("eu.ascens.helena.CompAttrGetterLogger");

	private final String fieldName;
	private final Class<T> type;

	public CompAttrGetter(String fieldName, Class<T> type) {
		this.fieldName = fieldName;
		this.type = type;
	}

	@Override
	Set<Component> lockObjects(Role source) {
		return Auxiliaries.getAsSet(source.getOwner());
	}

	/**
	 * This method evaluates the value of the component attribute of the owner
	 * of the source role.
	 *
	 * @param source the role for which the component attribute of its owner
	 *            should be evaluated
	 * @return
	 * @throws PropertyNotDeclaredInClassException This exception is thrown if
	 *             the guard evaluates a property which is not defined in its
	 *             base class.
	 * @throws ReflectionException This exception is thrown if the guard
	 *             evaluates a property which could not be retrieved via
	 *             reflection.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public T getValue(Role source)
			throws PropertyNotDeclaredInClassException, ReflectionException {

		DataFieldType attrType = DataFieldType.getType(this.fieldName);
		ComponentType compType = source.getOwner().getType();

		if (!compType.isAllowed(attrType)
				|| (attrType.getType() != this.type)) {
			throw new PropertyNotDeclaredInClassException(this.fieldName,
					source.getOwner().getClass());

		}

		Object value = source.getOwner().getCompAttr(this.fieldName);

		this.log.fine("Role " + source.getType()
				+ " retrieved component attribute " + this.fieldName);

		return (T) value;
	}
}
