! $RCSfile: data_lightning.f90,v $
! $Revision: 1.0 $ $Date: 2009/02/25 05:50:55 $
!+ Variables for the computation of the lightning frequency
!------------------------------------------------------------------------------

MODULE data_lightning

!------------------------------------------------------------------------------
!
! Description:
!  This data module contains data necessary for the computation of
!  thr lightning frequency.
!
! Current Code Owner: DLR, Johannes Dahl
!  phone:  +49  8153 28 3556
!  email:  johannes.dahl@dlr.de
!
! History:
! Version    Date       Name
! ---------- ---------- ----
! 1.0        2010/01/26 Johannes Dahl
!  Initial release
!
! Code Description:
! Language: Fortran 90.
! Software Standards: "European Standards for Writing and
! Documenting Exchangeable Fortran 90 Code".
!==============================================================================
    
USE data_parameters , ONLY :  &
    ireals,    &  ! KIND-type parameters for real variables
    iintegers     ! kind-type parameter for "normal" integer variables

!==============================================================================

IMPLICIT NONE

!==============================================================================

!------------------------------------------------------------------------------
! COSMO-DE correction factors
!------------------------------------------------------------------------------

! COSMO-DE-specific correction factors that account for the unrealistic
! horizontal size of the storms, as well as for the slightly too low
! graupel mass (based on the investigated cases).  These factors
! should strive towards 1.0 as the model captures the correct size
! and hydrometeor distribution.

REAL (KIND=ireals), PARAMETER ::         &
  graupel_mass_correction = 1.2_ireals,  &
  storm_width_correction  = 0.25_ireals

!------------------------------------------------------------------------------
! Section 1: global structures
!------------------------------------------------------------------------------

! All x-, y-indices are global 

! Structure containing information on analyzed clusters
! -----------------------------------------------------

TYPE cluster_info
  INTEGER (KIND=iintegers) ::   label
  INTEGER (KIND=iintegers) ::   pixels
  INTEGER (KIND=iintegers) ::   cent_pos_x
  INTEGER (KIND=iintegers) ::   cent_pos_y
  INTEGER (KIND=iintegers) ::   cent_pos_z
  INTEGER (KIND=iintegers) ::   vert_extent
  INTEGER (KIND=iintegers) ::   top_x 
  INTEGER (KIND=iintegers) ::   top_y                              !   -
  INTEGER (KIND=iintegers) ::   top_z                              !   - 
  REAL    (KIND=ireals)    ::   bottom_height                      ! [m  ]
  REAL    (KIND=ireals)    ::   top_height                         ! [m  ]
  REAL    (KIND=ireals)    ::   average                            ! [depends]
END TYPE cluster_info

! Structure containing relevant/final cell info
! ---------------------------------------------

TYPE storm
  INTEGER (KIND=iintegers) ::  centroid_x                          ! [1  ]
  INTEGER (KIND=iintegers) ::  centroid_y                          ! [1  ]
  REAL    (KIND=ireals)    ::  flash_rate                          ! [1/s]
END TYPE storm

! 2-class cells (capacitor structure: upper and lower plate)
! ----------------------------------------------------------

TYPE capacitor
  INTEGER (KIND=iintegers) ::   cap_label 

  ! upper plate:
 
  INTEGER (KIND=iintegers) ::   top_label 
  INTEGER (KIND=iintegers) ::   top_pixels         
  REAL    (KIND=ireals)    ::   top_area             ! [m^2]
  REAL    (KIND=ireals)    ::   top_depth            ! [m  ]
  INTEGER (KIND=iintegers) ::   top_cent_z

  ! capacitor properties

  REAL    (KIND=ireals)    ::   total_height         ! [m  ]
  REAL    (KIND=ireals)    ::   separation           ! [m  ] SFC distance
  REAL    (KIND=ireals)    ::   plate_distance       ! [m  ] center distance 
                                                          (or 2D plate dist) 
  REAL    (KIND=ireals)    ::   breakdown_alt        ! [m] between plates 
  INTEGER (KIND=iintegers) ::   x_pos                ! horizontal coordinates
  INTEGER (KIND=iintegers) ::   y_pos                                     
  
  ! lower plate:
  
  INTEGER (KIND=iintegers) ::   bot_label
  INTEGER (KIND=iintegers) ::   bot_pixels
  REAL    (KIND=ireals)    ::   bot_area                ! [m^2]
  REAL    (KIND=ireals)    ::   bot_depth               ! [m  ]
  INTEGER (KIND=iintegers) ::   bot_cent_z

  REAL    (KIND=ireals)    ::   graupel                 ! [kg/kg]  
  REAL    (KIND=ireals)    ::   riming                  ! [kg/kg/s] 
 
END TYPE capacitor

!-----------------------------------------------------------------------------
! Utility structure containing some information on updraft positions
! and their area (area + loc)
!-----------------------------------------------------------------------------

TYPE area_loc
  INTEGER (KIND=iintegers) :: cloc_x    ! centroid coordinates (global)
  INTEGER (KIND=iintegers) :: cloc_y    ! 
  INTEGER (KIND=iintegers) :: cloc_z    ! 
  REAL    (KIND=ireals)    :: area      ! cluster area
  INTEGER (KIND=iintegers) :: amount    ! number of clusters
END TYPE area_loc

!-----------------------------------------------------------------------------
! Section 2: Global scalars
!-----------------------------------------------------------------------------

INTEGER (KIND=iintegers)  ::  &
  cs_count,                   &  ! number of CBs (length of final csize)
                                 ! this is just the number of nontrivial 
                                 ! elements (either length of CSIZE or number 
                                 ! of overlapping capacitor plates if 
                                 ! itype_lightning == 1; assigned in LABEL 
                                 ! and re-assigned in DISTRIBUTE_FLASHES
                                 ! routines) 

  number_overlaps                ! Number of vertically overlapping clusters
                                 ! assigned in CLUSTER_OVERLAPS routine 

! global parameters
! -----------------
!
! Remarks:
! --------
!
! The most general definition would be:
! csize_length           = ie_tot * je_tot * ke_tot / 2_iintegers
!                        = 4852025_iintegers
! n_overlapping_elements = ke_tot * ie_tot 
!                        = 21050_iintegers
!
! This would result in large memory allocation, which is not ever needed
! for the current application.  The following assignments may be adjusted
! as required by the user.  They are still selected to be too large,
! to be on the safe side.

INTEGER (KIND=iintegers), PARAMETER   ::      &
  csize_length           = 1000000_iintegers, &  ! Length of global csize array 
  n_overlapping_elements = 100000_iintegers      ! max allowed number of 
                                                 ! overlapping clusters (x2)


REAL (KIND=ireals)  ::  &
  qg_boundary = 1.0_ireals         ! QC value in g/kg defining the lateral edge
                                   ! of a thunderstorm cloud (needs to overlap
                                   ! with a region of qg > qg_binmar 

! Automatic global derived types
! ------------------------------

INTEGER (KIND=iintegers), PARAMETER :: &
  max_number_cap = 5000_iintegers        ! max number of CBs

TYPE(capacitor) :: &
  info_cap(max_number_cap) 

!==============================================================================
! End module data_lightning
!==============================================================================

END MODULE data_lightning
