! $RCSfile: src_lightning.f90,v $
! $Revision: 1.0 $ $Date: 2008/03/18 10:12:15 $
!+ Source Module for generation of lightning output 
!----------------------------------------------------------------------------

MODULE src_lightning

!----------------------------------------------------------------------------
!
! Description:
!
!   This module procedure  calculates the bulk lightning frequency of 
!   individual thunderstorm cells.  Based on the instantaneous flash rate, 
!   the number of flashes for every cell between two calls of the lightning 
!   scheme is determined.  These flashes are distributed around the cells' 
!   centroid  positions.  Output is ASCII files containing the (x,y,t) 
!   coordinates of each discharge.  In addition, information about each
!   thunderstorm cell may optionally be printed to the log file. 
!
! The following parameterizations may be selected:
!
! itype_light == 1:
!   Parameterization based on the rate of increase with time of the
!   electrostatic field as well as on the neutralization efficiency
!   (Dahl's approach; subroutine dahl_2010).
!
! itype_light == 2:
!   Price and Rind parameterization (1992, JGR; subroutine pr92_updraft), 
!   where the height of the cell is determined from the vertical-velocity 
!   (updraft) distribution. 
!
! itype_light == 3:
!   Price and Rind parameterization (1992, JGR; subroutine pr92_cwi), 
!   where the height of the cell is determined from the hydrometeor 
!   (qc, qi) distribution.
!   This parameterization is retained for research purposes and 
!   does not yield good results because stratiform clouds are 
!   erroneously identified as thunderstorms.  Also, the cells' centroid
!   positions cannot be determined if the share a common anvil cloud. 
!
! itype_light == 4:
!   Parameterization by Yoshida et al. (2009, JGR; subroutine ymuk_2009).  
!   The cold cloud depth is determined by the updraft distribution 
!   (w > 2 m/s).
!
! itype_light == 5: Grewe et al. (2001; subroutine getal_2001) scheme
!   which is a re-formulated version of PR92.  The vertical-velocity
!   field is used to define updraft depth.
!    
! Method:
! 
!  In essence, the bulk of the parameterization is 3D cluster statistics,
!  i.e. contiguous regions where certain criteria are fulfilled (certain 
!  updraft velocities, riming rates, etc.) have to be identified and 
!  analyzed.  The size and the distribution of these clusters determine the 
!  lightning frequency.
!
!  Current Code Owner: DLR, Johannes Dahl 
!  phone:  +49 (0)8153 28 3556
!  email:  Johannes.Dahl@dlr.de 
!       
! History:
! Version    Date       Name
! ---------- ---------- -------------
! 1.1        2010/01/26 Johannes Dahl
!  Initial release
!
! Code Description:
!   Language          :  Fortran 90.
!   Software Standards: "European Standards for Writing and
!                        Documenting Exchangeable Fortran 90 Code".
!
!==============================================================================

!------------------------------------------------------------------------------
! Modules used:
!------------------------------------------------------------------------------

USE data_fields  , ONLY :  &
   
! 1. constant fields for the reference atmosphere                     (unit)
! -----------------------------------------------

    rho0       ,    & ! reference density at the full model levels    (kg/m3)
    dp0        ,    & ! reference pressure thickness of layers        ( Pa)
    p0         ,    & ! reference pressure at main levels             ( Pa)
    hhl        ,    & ! geometical height of half model levels        ( m )

! 2. external parameter fields                                        (unit)
! ----------------------------

    rlat       ,    & ! geographical latitude                         ( rad )
    rlon       ,    & ! geographical longitude                        ( rad )
    fc         ,    & ! coriolis-parameter                            ( 1/s )
    crlat      ,    & ! cosine of transformed latitude
    acrlat     ,    & ! 1 / ( crlat * radius of the earth )           ( 1/m )

! 3. prognostic variables                                             (unit)
! -----------------------

    u          ,    & ! zonal wind speed                              ( m/s )
    v          ,    & ! meridional wind speed                         ( m/s )
    w          ,    & ! vertical wind speed (defined on half levels)  ( m/s )
    t          ,    & ! temperature                                   (  k  )
    qv         ,    & ! specific water vapor content                  (kg/kg)
    qc         ,    & ! specific cloud water content                  (kg/kg)
    qi         ,    & ! specific cloud ice   content                  (kg/kg)
    qr         ,    & ! specific rain water  content                  (kg/kg)
    qs         ,    & ! specific snow  ice   content                  (kg/kg)
    qg         ,    & ! specific graupel ice content                  (kg/kg)
    pp         ,    & ! deviation from the reference pressure         ( pa  )

! 4. tendency fields for the prognostic variables                     (unit )
! -----------------------------------------------
!    timely deviation  by diabatic and adiabatic processes
!    without sound-wave terms

    utens        ,  & ! u-tendency without sound-wave terms           ( m/s2)
    vtens        ,  & ! v-tendency without sound-wave terms           ( m/s2)
    wtens        ,  & ! w-tendency without sound-wave terms           ( m/s2
                      ! (defined on half levels )
    ttens        ,  & ! t-tendency without sound-wave terms           ( K/s )
    qvtens       ,  & ! qv-tendency                                   ( 1/s )
    qctens       ,  & ! qc-tendency                                   ( 1/s )
    qitens       ,  & ! qi-tendency                                   ( 1/s )
    pptens       ,  & ! pp-tendency without sound-wave terms          (Pa/s )

! 5. fields for surface values and soil model variables               (unit )
! -----------------------------------------------------

    ps        ,     & ! surface pressure                              ( pa  )
    rho       ,     & ! density of moist air

!   fields of the radiation
    sohr       ,   & ! rate of solar heating                          ( K/s )
    thhr       ,   & ! rate of thermal heating                        ( K/s )

!   fields of the precipitation
    qrs        ,    & ! precipitation water (water loading)           (kg/kg)
    tt_conv    ,    & ! temperature tendency due to convection        (K /s )
    qvt_conv   ,    & ! humidity    tendency due to convection        ( s^-1)
    qct_conv   ,    & ! qc-tendency tendency due to convection        ( s^-1)
    qit_conv   ,    & ! qi-tendency tendency due to convection        ( s^-1)
    ut_conv    ,    & ! u-tendency due to convection                  (ms^-2)
    vt_conv    ,    & ! v-tendency due to convection                  (ms^-2)

!   fields that are computed in the dynamics
    dqvdt             ! threedimensional moisture convergence         (  1/s)

! 6. fields for model output and diagnostics
!--------------------------------------------------
    
! none

!------------------------------------------------------------------------------
! end of data_fields
!------------------------------------------------------------------------------

USE data_parallel,   ONLY :  &
    my_cart_id,              &
    num_compute,             &
    imp_integers,            & 
    icomm_cart,              &
    my_world_id,             &
    icomm_world,             &
    imp_logical,             &
    nproc, intbuf,           &
    logbuf,                  &
    realbuf,                 &
    charbuf,                 &
    imp_character,           &
    imp_reals,               &
    nboundlines,             &
    ldatatypes,              &
    ncomm_type,              &               
    sendbuf,                 &
    isendbuflen,             &
    my_cart_neigh

!------------------------------------------------------------------------------

USE data_lightning,   ONLY :  &
     cluster_info,            & ! statistics of the individual cells   
     storm,                   & ! individual cell data (x, y, flash rate)
     area_loc,                & ! cropped w-cluster information
     cs_count,                & ! number of non-trivial elements of csize
                                ! (which is initialized to contain 8E4 elements
     capacitor,               & ! structure containing capacitor information
     csize_length,            &
     n_overlapping_elements,  & ! Total number of elements that belong
                                ! to overlapping clusters
     number_overlaps,         & ! number of overlapping "capacitor plates"
     info_cap,                & ! capacitor structure
     max_number_cap,          & ! maximum number of capacitor cells
     qg_boundary,             & ! lateral cloud boundary in terms of QC
     graupel_mass_correction, & ! graupel-mass correction
     storm_width_correction     ! storm-width corrction

!------------------------------------------------------------------------------

USE pp_utilities,     ONLY :  &
    calc_sdi

!------------------------------------------------------------------------------

USE utilities,        ONLY :  &
  phirot2phi, rlarot2rla

!------------------------------------------------------------------------------

USE data_parameters , ONLY :  &
    ireals,    & ! KIND-type parameters for real variables
    iintegers    ! kind-type parameter for "normal" integer variables

!------------------------------------------------------------------------------

USE data_modelconfig, ONLY :  &

! 1. horizontal and vertical sizes of the fields and related variables
! --------------------------------------------------------------------
    ie,           & ! number of grid points in zonal direction
    je,           & ! number of grid points in meridional direction
    ieje,         & ! ie * je
    ke,           & ! number of grid points in vertical direction
    ke1,          & ! ke+1
    ie_tot,       & ! total number of gridpoints in zonal direction
    je_tot,       & ! total number of gridpoints in meridional direction
    ke_tot,       & ! total number of gridpoints in vertical direction  

! 2. start- and end-indices for the computations in the horizontal layers
! -----------------------------------------------------------------------
!    These variables give the start- and the end-indices of the
!    forecast for the prognostic variables in a horizontal layer.
!    Note, that the indices for the wind-speeds u and v differ from
!    the other ones because of the use of the staggered Arakawa-C-grid.

    istartpar,    & ! start index for computations in the parallel program
    iendpar,      & ! end index for computations in the parallel program
    jstartpar,    & ! start index for computations in the parallel program
    jendpar,      & ! end index for computations in the parallel program
    jstart, jend, istart, iend, &

! 3. variables for the time discretization and related variables
! --------------------------------------------------------------
    dt,           & ! timestep
    dt2,          & ! 2 * dt

! 4. variables describing the domain configuration
! ------------------------------------------------

   startlon_tot, &  ! rotated longitude of the lower left grid point
                    ! of the total domain (in degrees, E>0)
   startlat_tot, &  !
   dlon,         &
   dlat,         &
   pollon,       & ! longitude of the rotated north pole (in degrees, E>0)
   pollat,       & ! latitude of the rotated north pole (in degrees, N>0)
   polgam          ! angle between the north poles of the systems

! end of data_modelconfig
!---------------------------------------------------------------------------------

USE data_constants  , ONLY :   &
    pi, r_earth            

!---------------------------------------------------------------------------------

USE data_runcontrol , ONLY :   &

! 1. start and end of the forecast
! --------------------------------
    ntstep,       & ! actual time step
    nstart,       & ! first time step of the forecast
    nold,         & ! corresponds to ntstep - 1
    nnow,         & ! corresponds to ntstep
    nnew,         & ! corresponds to ntstep + 1

! 2. controlling the physics
! --------------------------
    itype_gscp,      & ! type of grid-scale precipitation physics
    lprogprec,       & ! forecast with prognostic rain and snow (qr, qs)
    ldiniprec,       & ! diagnostic initialisation of prognostic precip (qr, qs)
    itype_light,     & ! type of lightning parameterization   
    lightning,       & ! switch if lightning package is used
    hinclight,       & ! increment in hours when lightning is called
    lightning_step,  & ! increment in time steps when lightning is called

! 3. controlling the dynamics
! ---------------------------
    l2tls,        & ! forecast with 2-TL integration scheme
    irunge_kutta, & ! type of 2-TL (Runge-Kutta) scheme

! 4. additional control variables
! --------------------------------
    ldiabf_lh,    & ! include diabatic forcing due to latent heat in RK-scheme

! 5. controlling verbosity of debug output
! -----------------------------------------
    idbg_level,   & ! to control the verbosity of debug output
    ldebug_gsp,   & ! if .TRUE., debug output for grid scale precipitation
    lprintdeb_all   ! .TRUE.:  all tasks print debug output
                    ! .FALSE.: only task 0 prints debug output

!------------------------------------------------------------------------------
! end of data_runcontrol
!------------------------------------------------------------------------------

USE environment,        ONLY : get_free_unit, release_unit, model_abort, &
                               exchg_boundaries

USE parallel_utilities, ONLY : i_global, j_global, global_values, ij_local   

!------------------------------------------------------------------------------
! Module variables
!------------------------------------------------------------------------------

IMPLICIT NONE

! 1D module arrays
! ----------------

INTEGER (KIND=iintegers), PRIVATE, ALLOCATABLE ::  &
  global_j(:)                     ! output fron i/j_global

INTEGER (KIND=iintegers), PRIVATE  ::  &
  zrs(max_number_cap)             ! Array containing vertical centroid 

!==============================================================================

INCLUDE "mpif.h"

!==============================================================================

!==============================================================================
! Module procedures
!==============================================================================

CONTAINS

!==============================================================================        

SUBROUTINE organize_lightning

!------------------------------------------------------------------------------
!
! Description:
!  
! This subroutine selects the parameterization scheme to calculate the lightning
! frequency, prescribed by the namelist input parameter itype_light.
!
! itype_light == 1: Scheme based on E-field build-up rate and field-
!                   neutralization efficiency (dahl_2010).
!
! itype_light == 2: PR92 parameterization, based on updraft speed
!                   GT 2 m/s to determine the cell depths (pr92_updraft)   
!                      
! itype_light == 3: PR92 parameterization, based on cloud-water/ice
!                   content to determine the cell depths (pr92_cwi)  
!
! itype_light == 4: Yoshida et al. (2009) scheme based on 
!                   f ~ P relationship (ymuk_2009).
!
! itype_light == 5: Grewe et al. (2001) scheme
!                   which is a re-formulated version of PR92 (getal_2001).
!-------------------------------------------------------------------------------

IMPLICIT NONE

!-------------------------------------------------------------------------------
! Start routine
!-------------------------------------------------------------------------------

! Initialize utility module variables
! -----------------------------------

ALLOCATE(global_j(je))

CALL lightning_utilities

! Choose parameterization based on namelist input
! -----------------------------------------------

IF ( lightning ) THEN
  IF     ( itype_light == 1 ) THEN
    ! Call Dahl, 2010, parameterization 
    CALL dahl_2010
  ELSEIF ( itype_light == 2 ) THEN 
    ! Call Price and Rind, 1992, parameterization (updraft based)
    CALL pr92_updraft
  ELSEIF ( itype_light == 3) THEN 
    ! Call Price and Rind, 1992, parameterization (hydrometeor based)
    CALL pr92_cwi  
  ELSEIF ( itype_light == 4) THEN
    ! Call Yoshida et al., 2009, power-based parameterization 
    CALL ymuk_2009
  ELSEIF ( itype_light == 5) THEN
    ! Call Grewe et al., 2001, vertical-velocity based parameterization  
    CALL getal_2001
  ENDIF
ENDIF

DEALLOCATE(global_j)

!-------------------------------------------------------------------------------
! End module procedure organize_lightning
!-------------------------------------------------------------------------------

END SUBROUTINE organize_lightning

!===============================================================================
! Subroutine dahl_2010
!===============================================================================

SUBROUTINE dahl_2010

!-------------------------------------------------------------------------------
! Description:
!  This routine calculates the flash rate based on the build-up rate of the 
!  electrostatic field as well as the "neutralization efficiency" of the
!  flashes.  Prescribed are the sedimentation velocity which is set to vary
!  with the graupel mass, Mg. Also, the charging-current density varies with Mg.
!  The flash charge is varying with the volume of the space charges, based
!  on laboratory streamer propagation experiments with PMMA (Cooke, Williams,
!  late 70s, early 80s).
!
! Method:
!  Cluster analysis is performed to obtain the geometric term.  Then, the
!  critical charge is calculated, and the charge deposited in the flash is
!  determined.  With this, the neutralization efficiency can be determined.  
!  This and the given/prescribed distributions of charge transfer and 
!  charging-current density are inserted into the expression for the flash 
!  rate.  DISTRIBUTE_FLASHES then positions
!  the discharges underneath and around the thunderstorm cell.
!-------------------------------------------------------------------------------

IMPLICIT NONE

! Input arguments
! --------------

! None

! Output arguments
! ----------------

! None

!-------------------------------------------------------------------------------
! Local variables
!-------------------------------------------------------------------------------

! Derived types
! -------------

TYPE(storm), ALLOCATABLE :: &
  cell(:)

TYPE(area_loc), ALLOCATABLE ::  &
  w_area_loc(:)

! Local 3D-arrays
! ---------------

REAL (KIND=ireals)       ::  &
  qs3d(ie,je,ke),            &      ! 3D (diagnostic) versions of qx
  qi3d(ie,je,ke),            &
  qc3d(ie,je,ke),            &
  qci(ie,je,ke),             &
  csi (ie,je,ke),            &
  cwi (ie,je,ke),            &
  qsi(ie,je,ke),             &
  mg3d(ie,je,ke),            &
  wfl(ie,je,ke)                    ! w on full levels
  

INTEGER (KIND=iintegers) ::  &
  csi_bin(ie,je,ke),         &     ! Binary fields as input for LABEL 
  mg_bin(ie,je,ke),          &
  mg_bin_lab(ie,je,ke),      &
  qsi_bin(ie,je,ke),         &
  w_binary(ie,je,ke),        &
  w_labeled(ie,je,ke)

! Local 2D-arrays
! ---------------

! None

! Local 1D-arrays
! ---------------

REAL (KIND=ireals) ::        &
  gr_diam   (100),           &     ! Parameters and parameterized variables
  v_gr      (100),           &
  rho_charge(100),           &
  delta_q   (100),           &
  v_range   (100),           &
  grar_range(100),           &
  mg_range  (100) 

INTEGER (KIND=iintegers) ::  &
  n_array(100),              &
  wcsize(csize_length)

! Local dynamic 1D-arrays
! -----------------------

REAL (KIND=ireals), ALLOCATABLE  ::  &
  fl_lon(:),                         &  ! lon/lat/t of discharges
  fl_lat(:),                         &
  fl_time(:),                        &
  cdist(:)

! Local scalars
! -------------

INTEGER (KIND=iintegers) :: &
  i, j, k, ii, jj, kk,      &   ! loop variables
  istat, ierror,            &   ! status/error flags
  accum_flashes,            &
  array_count,              &
  mg_index,                 &
  vol_index,                &
  delta_x, delta_y,         &
  cindex,                   &
  n_clusters                   ! number of w-regions


REAL (KIND=ireals)  :: &
  plate_depth,         &  ! Mean of upper and lower plate depths
  plate_dist,          &
  geometric_term_num,  &
  geometric_term_den,  &
  geometric_term,      &
  sigma_crit,          &
  area,                &  ! area of the capacitor plates
  plate_strength,      &  ! GRAR or QG, determining charge density        
  sgr_diam,            &
  sv_gr,               &
  srho_charge,         &
  charge_volume,       &
  sdelta_q,            &
  e_crit,              & ! e_crit 
  alt_be,              & ! Breakeven altitude in m
  q_crit,              & ! amount of charge required for E_crit
  rho_crit,            & ! respective 3D chage density  
  j_gen,               & ! generator current density
  charge_after_flash,  & ! 
  sigma_after_flash,   & ! 2D charge density after the flash
  efac,                & ! factor for E-field expression
  e_after_flash,       & ! E-field after flash
  efne,                & ! E-Field Neutralization Efficiency
  crme,                & ! Charge Removal Efficiency 
  sigma_efficiency,    & ! same but w.r.t. sigma
  f_rate_term1,        & ! factors of flash-rate expression
  f_rate_term2,        & !
  lf_dl10,             & ! flash rate 
  min_cdist,           & !
  volume_corr,         & ! Function to account for too large graupel regions
  graupel_mass_corr,   & ! Fct accounting for too small graupel mass
  diameter,            & ! graupel-region's equivalent circular diameter      
  mcs_diameter,        & ! Diameter beyond which storm is MCS (def 15 km)
  gauss_width,         & ! Describes decay rate towards 0.4 from 1.0
  mcs_correction,      & ! Correction factor [0.4, 1.0]
  lf_dl10_uc,          & ! non-MCS-corrected flash rate (too large)
  gauss_arg              ! Exponent of Gauss function

! Local parameters
! ----------------

REAL (KIND=ireals), PARAMETER ::   &
  tcr         = 263.0_ireals,      & ! Charge-reversal temperature  
  eps         = 8.854E-12_ireals,  & ! Permittivity of air
  gamma       = 0.9_ireals,        & ! how much lightning contributes to disch
  facgrdiam   = 1.5873E-4_ireals,  &
  facgr       = 422.0_ireals,      &  
  facrhoc     = 1.4E-10_ireals,    & ! slope of grar-rho_ch relationship
  facqg       = 0.049_ireals,      & 
  vfac_grar   = 6.0E9_ireals,      &
  vfac_qg     = 12.0E9_ireals,     &
  efac1       = -201.736_ireals,   &
  efac2       = 1.E3 * 8.4_ireals    ! ALT in km

! Thresholds for labeling routine 

REAL (KIND=ireals), PARAMETER ::  &
  qsi_binmar  = 0.1_ireals,       & 
  grar_binmar = 0.1_ireals,       &
  mg_binmar   = 0.1_ireals,       &
  w_binmar    = 2.0_ireals

! Utility fields for debugging
! ----------------------------

REAL (KIND=ireals) ::      &
  tmark1, tmark2, deltat1, &         ! time stamp
  tmark3, tmark4, deltat2

INTEGER (KIND=iintegers)  ::  &
  ierr, countt

! Logical variables
! -----------------

LOGICAL, PARAMETER  ::   &
  lverbose     = .TRUE.       ! Print info

!==============================================================================
! Start routine
!==============================================================================

!------------------------------------------------------------------------------
! Section 1: THE PARAMETER - VARIABLES
!            
!            Define how terminal fall velocity of graupel, space-charge
!            density in the charging current, and the charge transferred in a
!            flash (i.e., the parameterized variables) vary with 
!            the parameters.
!
! NOTE:
!  In this implementation, a lookup-table style is used; a more efficient
!  description is:
!
!  delta_q(V) = 25 * (1 - EXP(0.067 - 0.027 * V)), where V is the charge 
!                                                  volume in km^3
!  
!  rho_ch(mg) = 4.467E-10 + 3.067E-9 * mg, where mg is the graupel mass
!					   in g/m^3	
!  diam(mg)   = 1.833E-3 + 3.333E-3 * mg
!
!  If mg GT 3 g/m^3, then rho_ch(mg=3) and diam(mg=3) are to be used.
!------------------------------------------------------------------------------

!------------------------------------------------------------------------------
! Section 1.1.: THE PARAMETERS
! Set up linearly-increasing arrays that contain possible ranges of parameters
! used to determine charging current, channel-propagation depths, and the like.
!------------------------------------------------------------------------------

! Create array containing "independent variables"
! -----------------------------------------------

array_count = 1

DO i = 1, 100
  n_array(i)  = array_count
  array_count = array_count + 1
ENDDO

! Range of graupel content
! ------------------------

mg_range =  0.5_ireals * (0.1_ireals + 0.06_ireals * n_array)

! Range of volume
! ---------------

! volume ranges from [2.5, 297.025] km^3

 v_range = 1.0E9_ireals * (2.5_ireals + 2.975_ireals * (n_array - 1)) 

!------------------------------------------------------------------------------
! Section 1.2: THE PARAMETERIZED VARIABLES
!
!   Define how variables used by the flash-rate equation depend on parameters
!   defined in Sec. 1.1.
!------------------------------------------------------------------------------

gr_diam = 0.002_ireals + 0.0001_ireals * n_array 

v_gr    = facgr * EXP(0.89_ireals * LOG(gr_diam))

! charge density in the charging current in C/m3
! ----------------------------------------------

rho_charge = &
2.0_ireals * 0.3E-9 + 2.0_ireals * (5.0E-9 - 0.4E-9) / 100. * n_array

! charge per flash in C
! ---------------------

delta_q = 25.0_ireals * (1.0_ireals - EXP(-0.08_ireals * n_array))

!-------------------------------------------------------------------------------
! Section 2: Define thunderstorm cells
!-------------------------------------------------------------------------------

! Prepare upper region (positive charge region)
! ---------------------------------------------

qc3d(:,:,:) = qc(:,:,:,nnow)
qs3d(:,:,:) = qs(:,:,:,nnow)
qi3d(:,:,:) = qi(:,:,:,nnow)
qci         = 1.0E3_ireals * qi3d + qc3d
qsi         = 1.0E3_ireals * (qs3d + qi3d)      ! in g/kg
qsi_bin     = BIN_FIELD(qsi, qsi_binmar)

! Prepare lower region (negative-charge region)
! ----------------------------------------------

mg3d(:,:,:) =  1.0E3_ireals * rho(:,:,:) * qg(:,:,:,nnow)  ! in g/kg

! Only consider that part of the field where T < -13 C (charge-reversal)

WHERE (t(:,:,:,nnow) > tcr) mg3d(:,:,:) = 0.0_ireals
mg_bin = BIN_FIELD(mg3d, mg_binmar)

!-------------------------------------------------------------------------------
! Section 2.1: Search storm cells
!   In the subroutine cluster_overlaps, the capacitor_details routine is called
!   which updates global variables that are used farther below (like
!   number_overlaps).
!-------------------------------------------------------------------------------

CALL cluster_overlaps (qsi_bin, mg_bin, mg_bin_lab)

!IF (number_overlaps > 0) THEN  
!  DO ii = 1, number_overlaps
!    print *, my_cart_id, 'timestep: ', ntstep 
!    print *, my_cart_id, 'CAPACITOR INFO', ii,   &
!    INT(info_cap(ii) % top_label),         &
!    INT(info_cap(ii) % top_pixels),        &
!    INT(info_cap(ii) % top_area * 1.0E-6), &
!    INT(info_cap(ii) % top_depth),         &
!    INT(info_cap(ii) % separation),        &
!    INT(info_cap(ii) % x_pos),             &
!    INT(info_cap(ii) % y_pos),             &
!    INT(info_cap(ii) % bot_label),         &
!    INT(info_cap(ii) % bot_pixels),        &
!    INT(info_cap(ii) % bot_area * 1.0E-6), &
!    INT(info_cap(ii) % bot_depth)
!  ENDDO
!ELSE IF (number_overlaps == 0) THEN
!  print *, my_cart_id, 'NOVEMBER: No cluster overlaps!', ntstep
!ENDIF

!----------------------------------------------------------------------------
! Section 3: Calculate the flash frequency
!----------------------------------------------------------------------------

! Some exception handling

IF (my_cart_id == 0 .AND. number_overlaps > max_number_cap) THEN
  WRITE (*,*) '     *** SRC_LIGHTNING.DAHL_2010 ERROR ***  '
  WRITE (*,*) 'SET max_number_cap VARIABLE TO AT LEAST', max_number_cap
ENDIF

ALLOCATE (cell(number_overlaps), STAT=istat)

cell % flash_rate = 0.0_ireals
cell % centroid_x = 0_iintegers
cell % centroid_y = 0_iintegers

IF (number_overlaps > 0) THEN

  DO ii = 1, number_overlaps
      
    plate_depth = &
    0.5_ireals * (info_cap(ii) % top_depth + info_cap(ii) % bot_depth)

    plate_dist = info_cap(ii) % plate_distance 

    area = info_cap(ii) % bot_area

    !----------------------------------------------------------------------
    ! Section 3.1:
    !   Determine variables that are related MG:
    !   graupel diameter (sgr_diam), sedimentation velocity (sv_gr), and
    !   charge density in generator current (srho_charge)
    !----------------------------------------------------------------------

    plate_strength = 1.0E3_ireals * info_cap(ii) % graupel
      
    IF (plate_strength >= MAXVAL(mg_range)) THEN
      mg_index = 100
    ELSE
      mg_index = 1
      mg_loop:  DO jj = 1, 100
        IF (mg_range(jj) >= plate_strength) THEN
          mg_index = jj
          EXIT mg_loop
        ENDIF
      ENDDO mg_loop
    ENDIF

    sgr_diam    = gr_diam   (mg_index)
    sv_gr       = v_gr      (mg_index)  
    srho_charge = rho_charge(mg_index)

    !----------------------------------------------------------------------     
    ! Section 3.3: Set up variables related to space-charge volume
    !----------------------------------------------------------------------

    charge_volume = plate_depth * area

    IF (charge_volume >= MAXVAL(v_range)) THEN
      vol_index = 100
    ELSE
      vol_index = 1
      vol_loop:  DO jj = 1, 100
        IF (v_range(jj) >= charge_volume) THEN
          vol_index = jj
          EXIT vol_loop
        ENDIF
      ENDDO vol_loop
    ENDIF 
     
    sdelta_q = delta_q(vol_index)  

    !----------------------------------------------------------------------
    ! Section 3.4: Calculate lightning frequency
    !----------------------------------------------------------------------  

    ! Breakeven field strength 
    ! ------------------------ 

    alt_be = info_cap(ii) % breakdown_alt  
    e_crit = 1.0E3_ireals * efac1 * EXP(-alt_be / efac2)   

    ! Geometric term
    ! --------------

    geometric_term_num = plate_dist
    geometric_term_den = &
    SQRT(area / pi + (0.5_ireals * plate_dist) * (0.5_ireals * plate_dist)) 

    geometric_term = geometric_term_num / geometric_term_den

    ! Critical charge (densities)
    ! -------------------------------------

    sigma_crit = 2.0_ireals * eps / (geometric_term - 2.0_ireals) * e_crit                    

    q_crit     = sigma_crit * area 
    rho_crit   = q_crit / (area * plate_depth) 

    ! Generator current
    ! -----------------

    j_gen = srho_charge * sv_gr         

    ! Make sure no more charge is transferred than is present initially

    IF (sdelta_q > q_crit) THEN
      sdelta_q = q_crit
    ENDIF
 
    charge_after_flash = q_crit - sdelta_q
    sigma_after_flash  = charge_after_flash / area

    ! E-field after the flash
    ! -----------------------  
 
    efac = 0.5 * sigma_after_flash / eps
    e_after_flash = -sigma_after_flash / eps + efac * geometric_term

    ! E-field-neutralization and charge-removal efficiencies (efne, crme)
    ! -------------------------------------------------------------------

    efne = (e_crit - e_after_flash) / e_crit
    crme = (q_crit - charge_after_flash) / q_crit
      
    sigma_efficiency = (sigma_crit - sigma_after_flash) / sigma_crit

    ! Calculate flash rate
    ! --------------------

    f_rate_term1 = (geometric_term / (2.0_ireals * eps) - &
                   1.0_ireals/eps) / e_crit
    f_rate_term2 = j_gen / efne
    lf_dl10_uc   = gamma * f_rate_term1 * f_rate_term2  ! MCS-uncorrected

    ! Apply MCS-correction (flash rate reduced by up to 60 %)
    ! -------------------------------------------------------

    diameter     = 2.0E-3_ireals * SQRT(area / pi)  ! in km
    mcs_diameter = 15.0_ireals                      ! in km
    gauss_width  = 0.085_ireals

    IF (diameter    <= mcs_diameter) THEN 
      mcs_correction = 1.0_ireals
    ELSEIF (diameter > mcs_diameter) THEN 
      gauss_arg      = gauss_width * (diameter - mcs_diameter)
      mcs_correction = 0.4_ireals + 0.6_ireals * EXP(-gauss_arg * gauss_arg)
    ENDIF

    lf_dl10 = mcs_correction * lf_dl10_uc 

    cell(ii) % flash_rate = lf_dl10

    !----------------------------------------------------------------------
    ! Section 3.5: Print thunderstorm cell information
    !----------------------------------------------------------------------

    IF (lverbose .AND. my_cart_id == 0) THEN     
      print *, '****************************************************'
      print *, '     SRC_LIGHTNING.DAHL_2010: Verbose output'
      print *, '****************************************************'
      print *, ''     
      print '(a35, i7, a25, i7, f7.2)', 'Max number of cells:',          &
                 number_overlaps, 'at time step/hour:',                    &
                 ntstep, ntstep * dt / 3600.0_ireals 
      print '(a35, i5)', 'Details of storm (label):', info_cap(ii) % bot_label 
      print *, ''
      print *, '             GEOMETRIC PARAMETERS:'
      print *, '             ---------------------'
      print '(a35, i6, a1, i4)', 'Position (X,Y):',    &
                info_cap(ii) % x_pos, ',',  info_cap(ii) % y_pos 
      print '(a35, f6.1, a6)', 'Equivalent circular diameter:',    &
      2.0E-3_ireals * SQRT(area / pi),                ' km'
      print '(a35, f6.1, a6)', 'Centroid distance:',    &
      1.0E-3_ireals * plate_dist, ' km'
      print '(a35, f6.1, a6)', 'Plate SFC distance:',    &
            1.0E-3_ireals * info_cap(ii) % separation,         ' km' 
      print '(a35, f6.1, a6)', 'Plate depths:',    & 
                1.0E-3_ireals * plate_depth,                   ' km' 
      print '(a35, f6.1, a6)', 'Storm-top height:',    &
            1.0E-3_ireals * info_cap(ii) % total_height,       ' km' 
      print '(a35, f6.1, a6)', 'Total storm depth:',    &
          1.0E-3_ireals*(info_cap(ii)%separation+2.0*plate_depth), ' km'
      print '(a35, f6.1, a6)', 'Charge volume:',    &
              1.0E-9_ireals * charge_volume,                   ' km**3' 
      print '(a35, f6.1, a8)', 'Max graupel content (QG)', &
          plate_strength, 'g/kg'
      print '(a35, f8.4)', 'Geometric term', geometric_term
      print *, '' 
      print *, '              ELECTRIC PARAMETERS:'
      print *, '             ---------------------'
      print '(a35, f6.1, a6)', 'Critical field:',    & 
                1.0E-3_ireals * e_crit,                            ' kV/m'
      print '(a35, f6.1, a6)', 'Total charge per plate:', q_crit,  ' C'
      print '(a35, e12.3, a8)', 'Charge per volume: ', rho_crit,    'C/m**3' 
      print '(a35, e12.3, a8)', 'Charge per area:', sigma_crit,     'C/m**2'
      print '(a35, f6.3, a6)', 'Graupel diameter:', 100.0 * sgr_diam, ' cm'
      print '(a35, f6.2, a6)', 'Sedimentation velocity:', sv_gr,    ' m/s'
      print '(a35, e12.3, a8)', 'Charging current density:', srho_charge, 'C/m**3'
      print '(a35, f6.1, a6)', 'Removed charge:',          &
                q_crit - charge_after_flash,                       ' C'
      print '(a35, f6.1, a6)', 'Removed E-field:',          &
                1.0E-3_ireals * (e_crit - e_after_flash),          ' kV/m' 
      print '(a35, f6.3)', 'Charge removal efficiency:', crme             
      print '(a35, f6.3)', 'Field neutralization efficiency:', efne
      print '(a35, f8.4, a10)', 'Uncorrected flash rate:', &
                lf_dl10_uc * 60.0_ireals,                          ' 1/min'
      print '(a35, f8.4, a10)', 'MCS-correction:', mcs_correction
      print '(a35, f8.4, a10)', 'Flash rate:', &
                lf_dl10 * 60.0_ireals,                             ' 1/min'
      print *, ''
    ENDIF
                            
    cell(ii) % centroid_x = info_cap(ii) % x_pos
    cell(ii) % centroid_y = info_cap(ii) % y_pos

  ENDDO
ELSE IF (number_overlaps == 0) THEN
  lf_dl10 = 0.0_ireals
ENDIF

!----------------------------------------------------------------------------
! Section 4: Distribute flashes around the centroid (space and time)
!----------------------------------------------------------------------------

accum_flashes = 0_iintegers

DO jj = 1, number_overlaps
  
  accum_flashes = accum_flashes + &
  NINT( cell(jj) % flash_rate * 3600.0_ireals * hinclight )

ENDDO

WRITE (*,'(f4.1, a25, i6)') hinclight * 60., &
            '-min accumulated flashes:', accum_flashes                             

IF (accum_flashes > 0) THEN
  ALLOCATE(fl_lon (accum_flashes), STAT = istat )
  ALLOCATE(fl_lat (accum_flashes), STAT = istat )
  ALLOCATE(fl_time(accum_flashes), STAT = istat )
ELSEIF (accum_flashes == 0) THEN
  ALLOCATE(fl_lon (2), STAT = istat )
  ALLOCATE(fl_lat (2), STAT = istat )
  ALLOCATE(fl_time(2), STAT = istat )
ENDIF

CALL distribute_flashes (cell, accum_flashes, fl_lon, fl_lat, &
                         fl_time, mg_bin_lab)

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(w_area_loc, STAT = istat)
DEALLOCATE(cdist,      STAT = istat)
DEALLOCATE(cell,       STAT = istat)
DEALLOCATE(fl_lon,     STAT = istat)
DEALLOCATE(fl_lat,     STAT = istat)
DEALLOCATE(fl_time,    STAT = istat)

!===============================================================================
! End subroutine DAHL_2010
!===============================================================================

END subroutine dahl_2010

!===============================================================================
! Subroutine pr92_updraft
!===============================================================================

SUBROUTINE pr92_updraft

!-------------------------------------------------------------------------------
! Description:
!  This module procedure calculates the stroke frequency based on the parameter-
!  ization by Price and Rind (JGR, 1992).
!
! Method:
!  Hoshen and Copleman's cluster identification algorithm, performed by the
!  subroutine LABEL.  Applied to the vertical-velocity field, the depth of the 
!  cloud can be calculated and the flash frequency be deduced. 
!------------------------------------------------------------------------------

IMPLICIT NONE

! Local scalars:
! -------------
  INTEGER (KIND=iintegers) ::  &
    k     ,            & ! loop index in vertical direction
    i     ,            & ! loop index in x-direction
    j     ,            & ! loop index in y-direction
    i_glob,            & ! From parallel utilities 
    j_glob,            &
    i_loc ,            &
    j_loc ,            &
    isubdomain,        &
    ierror, impierror, & ! Error flags 
    istat,             & ! allocation error flag  
    nt,                & ! Current time step  
    lun,               & ! Unit number for IO 
    ics_len,           & ! length of csize (unique entries)
    count, jj

  INTEGER (KIND=iintegers) ::  &
    accum_flashes        ! total number of flashes in hinclight period

  REAL (KIND=ireals)          ::  &
    storm_height        ! updraft top                 [m]

  ! Local parameters
  !-----------------

  REAL (KIND=ireals), PARAMETER ::  &
    prfac =  5.73333E-07,           &  ! 3.44E-5 / 60, factor for PR92 
                                       ! parameterization (but per second)
    prexp = 4.9_ireals                 ! power of "5th-power law"

  REAL (KIND=ireals), PARAMETER ::  &
    w_binmar = 2.0

  ! Local arrays
  !-------------

  REAL    (KIND=ireals   ) ::  &
    w_hfl(ie, je, ke)             ! de-staggered UVVs 

  INTEGER (KIND=iintegers) ::  &
    w_bin(ie, je, ke),         &
    w_lab(ie, je, ke),         &
    q_bin(ie, je, ke),         &  ! Filtered field
    q_lab(ie, je, ke)             ! Labeled (binary) field

  INTEGER (KIND=iintegers) ::  &
    icsize(1000000_iintegers)

  ! Allocatable arrays
  !-------------------

  REAL (KIND=ireals), ALLOCATABLE ::  &
    fl_lon (:),                       &
    fl_lat (:),                       &
    fl_time(:)                          

  TYPE(cluster_info), ALLOCATABLE ::  &
    w_clusters(:)

  TYPE(storm), ALLOCATABLE        ::  &
    cell(:)

  REAL (KIND=ireals), ALLOCATABLE ::  &
    lf_pr92_updraft(:)

  ! Logical variables
  ! -----------------

  LOGICAL                          :: &
    lverbose = .TRUE.

!=============================================================================
! Start program
!=============================================================================

!------------------------------------------------------------------------------
! Section 1.1:
!   Prepare vertical-velocity field for label procedure
!------------------------------------------------------------------------------

! Destagger w-field

w_hfl (:,:,1:ke) = 0.5 * (w(:,:,1:ke,nnow) + w(:,:,2:ke+1,nnow))

w_bin = bin_field(w_hfl, w_binmar)

!------------------------------------------------------------------------------
! Section 2.1:
!   Call labeling procedure and allocate arrays with cs_count
!   Calculate flash rate
!------------------------------------------------------------------------------

CALL label (w_bin, w_lab, icsize ) 

ics_len = cs_count

ALLOCATE(w_clusters(ics_len),      STAT = istat)
ALLOCATE(lf_pr92_updraft(ics_len), STAT = istat)
ALLOCATE(cell(ics_len),            STAT = istat)

! Initialize arrays/objects
! -------------------------

cell % flash_rate = 0.0_ireals
cell % centroid_x = 0_iintegers
cell % centroid_y = 0_iintegers

zrs               = 0_iintegers

DO jj = 1, ics_len
  zrs(jj) = 22_iintegers  ! Corresponds to breakdown-altitude of more 
ENDDO                     ! sophisticated scheme

!-----------------------------------------------------------------------------
! Calculate flash rate based on PR92; if cells are too small (less than 10
! gridpoints), flash frequency is set to zero 
!-----------------------------------------------------------------------------

IF (ics_len > 0) THEN
  CALL cluster_analysis (w_clusters, w_lab, icsize) 
  DO jj = 1, ics_len
    IF (w_clusters(jj) % pixels > 10) THEN   ! Reduce gravity-wave/Alps effect
      storm_height = 1.0E-3_ireals * (w_clusters(jj) % top_height)
      IF (storm_height > 0.0) THEN   ! to stay out of trouble with LOG function
        lf_pr92_updraft(jj) = &
        prfac * EXP(prexp * LOG(storm_height))         ! in s-1

        IF (lverbose .AND. my_cart_id == 0) THEN     
          print *, '****************************************************'
          print *, '   SRC_LIGHTNING.PR92_UPDRAFT: Verbose output'
          print *, '****************************************************'
          print *, ''     
          print '(a35, i7, a25, i7, f7.2)', 'Max number of cells:',          &
                   ics_len, 'at time step/hour:',                    &
                   ntstep, ntstep * dt / 3600.0_ireals 
          print '(a35, i5)', 'Details of storm (label):', jj
          print *, ''
          print '(a35, i6, a1, i4)', 'Position (X,Y):',    &
                  w_clusters(jj) % cent_pos_x, ',',  w_clusters(jj) % cent_pos_x 
          print '(a35, f8.1, a10)', 'Top height:',    &
          1.0E-3_ireals * w_clusters(jj) % top_height  , ' km'
          print '(a35, f8.4, a10)', 'Flash rate:',    &
          60.0_ireals * lf_pr92_updraft(jj),          '1/min'
          print *, ''
        ENDIF
 
      ELSE
        lf_pr92_updraft(jj) = 0.0_ireals
      ENDIF
    ELSE IF (w_clusters(jj) % pixels <= 10) THEN
      lf_pr92_updraft(jj) = 0.0_ireals
    ENDIF
  ENDDO
ELSE
  DO jj = 1, ics_len
    lf_pr92_updraft(jj) = 0.0_ireals
  ENDDO
ENDIF

!--------------------------
! Assign structure elements
!--------------------------

DO jj = 1, ics_len

  cell(jj) % centroid_x = w_clusters(jj) % cent_pos_x 
  cell(jj) % centroid_y = w_clusters(jj) % cent_pos_y
  cell(jj) % flash_rate = lf_pr92_updraft(jj) 

ENDDO

!--------------------------------------------------------
! Distribute flashes around the centroid (space and time)
!--------------------------------------------------------

accum_flashes = 0.0_iintegers

DO jj = 1, ics_len
  accum_flashes = accum_flashes + &
  NINT(lf_pr92_updraft(jj) * 3600.0_ireals * hinclight)
ENDDO

IF (accum_flashes > 0) THEN
  ALLOCATE(fl_lon (accum_flashes), STAT = istat )
  ALLOCATE(fl_lat (accum_flashes), STAT = istat )
  ALLOCATE(fl_time(accum_flashes), STAT = istat )
ELSEIF (accum_flashes == 0) THEN
  ALLOCATE(fl_lon (2), STAT = istat )
  ALLOCATE(fl_lat (2), STAT = istat )
  ALLOCATE(fl_time(2), STAT = istat )
ENDIF

CALL distribute_flashes (cell, accum_flashes, fl_lon, fl_lat, fl_time, w_lab)

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(w_clusters,      STAT = istat)
DEALLOCATE(lf_pr92_updraft, STAT = istat)
DEALLOCATE(cell,            STAT = istat)

DEALLOCATE(fl_lon,          STAT = istat)
DEALLOCATE(fl_lat,          STAT = istat)
DEALLOCATE(fl_time,         STAT = istat)

!============================================================================
! End subroutine pr92_updraft
!============================================================================

END SUBROUTINE pr92_updraft

!============================================================================
! Subroutine pr92_cwi
!============================================================================

SUBROUTINE pr92_cwi

!----------------------------------------------------------------------------
! Description:
!   This module procedure determines the flash frequency based on PR92.  To
!   determine the storm-top height, the upper boundary of the sum of cloud-
!   water and cloud ice (hence, the appended cwi) is used.
!   
! Method: 
!   The subroutine LABEL is called to find the Cb's, CLUSTER_ANALYSIS
!   determines the cell centroids as well as their depth, and 
!   DISTRIBUTE_FLASHES spreads the flash sites around the cells and writes
!   their times/positions into an ACII file.
!   Minimum cloud depth is set to 1 km, which corresponds to an optical
!   thickness (tau) of 23.  This should result in some errors, but it
!   corresponds to PR92.
!
! Note:
!   This routine is not recommended because all clouds that are thick enough
!   are considered to be electrified (including stratiform clouds).  Also,
!   flash distribution is unrealistic as, e.g., a large anvil may cover
!   several cells (say, n).  The flashes are distributed around the anvil 
!   centroid, rather than n times around the centroids of the cells.
!----------------------------------------------------------------------------

!----------------------------------------------------------------------------
! Declarations
!----------------------------------------------------------------------------

IMPLICIT NONE

! Subroutine arguments
!---------------------

! None

! Local scalars:
! -------------

  INTEGER (KIND=iintegers) ::  &
    k     ,            & ! loop index in vertical direction
    i     ,            & ! loop index in x-direction
    j     ,            & ! loop index in y-direction
    i_glob,            & ! From parallel utilities 
    j_glob,            &
    i_loc ,            &
    j_loc ,            &
    isubdomain,        &
    ierror, impierror, & ! Error flags 
    istat,             & ! allocation error flag  
    nt,                & ! Current time step  
    lun,               & ! Unit number for IO 
    ics_len,           & ! length of csize (unique entries)
    count, jj

  INTEGER (KIND=iintegers) ::  &
    accum_flashes        ! total number of flashes in hinclight period

  REAL (KIND=ireals)          ::  &
    cloud_top,                    &   ! cloud top                 [m]
    cloud_bottom,                 &   ! cloud bottom   
    cloud_depth, cloud_height

  ! Local parameters
  ! ----------------

  REAL (KIND=ireals), PARAMETER ::  &    
    prfac =  5.73333E-07,           &  ! 3.44E-5 / 60, factor for PR92 
                                       ! parameterization (but per second)
    prexp = 4.9_ireals                 ! power of "5th-power law"

  REAL       (KIND=ireals   ) ::  &
    cwi_binmar = 0.1_ireals

  ! Local arrays
  ! ------------

  REAL    (KIND=ireals   ) ::  &
    qc3d  (ie, je, ke),        &  ! 3D version of qc
    qi3d  (ie, je, ke),        &  ! 3D version of qi
    cwi   (ie, je, ke)            ! qc + qi

  INTEGER (KIND=iintegers) ::  &
    cwi_bin(ie, je, ke),       &  ! input for LABEL
    cwi_lab(ie, je, ke)           ! ourput of LABEL

  INTEGER (KIND=iintegers) ::  &
    icsize(1000000_iintegers)  

  ! Allocatable arrays
  ! ------------------

  REAL (KIND=ireals), ALLOCATABLE ::  &
    fl_lon (:),                       &
    fl_lat (:),                       &
    fl_time(:)                          

  TYPE(cluster_info), ALLOCATABLE ::  &
    cwi_clusters(:)

  TYPE(storm),   ALLOCATABLE ::  &
    cell(:)

  REAL (KIND=ireals), ALLOCATABLE ::  &
    lf_pr92_cwi(:)

! For debugging
! -------------

INTEGER (KIND=iintegers) ::             &
  global_field(ie_tot, je_tot, ke_tot), &
  global(ie_tot, je_tot, ke_tot)

INTEGER (KIND=iintegers) ::  &
  vals,                      &
  izerror,                   &
  iglob, jglob,              &
  ii  

!============================================================================
! Start procedure
!============================================================================

!------------------------------------------------------------------------------
! Section 1.1 (cwi):
!   Prepare qc and qi fields to determine the cloud boundaries
!------------------------------------------------------------------------------

qc3d(:,:,:) = qc(:,:,:,nnow)  
qi3d(:,:,:) = qi(:,:,:,nnow)

cwi = 1000.0_ireals * (qc3d + qi3d)

cwi_bin = bin_field(cwi, cwi_binmar)

!------------------------------------------------------------------------------
! Section 2.1 (cwi):
!   Call labeling procedure and allocate arrays with cs_count
!   Calculate flash rate
!------------------------------------------------------------------------------

CALL label (cwi_bin, cwi_lab, icsize ) 

ics_len = cs_count

ALLOCATE(cwi_clusters(ics_len), STAT = istat)
ALLOCATE(lf_pr92_cwi(ics_len),  STAT = istat)
ALLOCATE(cell(ics_len),         STAT = istat)

cell % flash_rate = 0.0_ireals
cell % centroid_x = 0_iintegers
cell % centroid_y = 0_iintegers
zrs               = 0_iintegers

DO jj = 1, ics_len
  zrs(jj) = 22_iintegers  ! Corresponds to breakdown-altitude of more 
ENDDO                     ! sophisticated scheme

!-----------------------------------------------------------------------------
! Section 3: (cwi)
!  Calculate flash rate based on PR92
!-----------------------------------------------------------------------------

lf_pr92_cwi = 0.0_ireals

IF (ics_len > 0) THEN
   
  CALL cluster_analysis (cwi_clusters, cwi_lab, icsize) 
  
  DO jj = 1, ics_len

    cloud_depth = &
    1.0E-3_ireals * (cwi_clusters(jj) % top_height - & 
                     cwi_clusters(jj)%bottom_height)  
    cloud_height = 1.0E-3_ireals * cwi_clusters(jj) % top_height
 
    IF (cloud_depth > 1.0) THEN   ! tau > 23 (PR92) corresponds to abt. 1 km 
      lf_pr92_cwi(jj) = &
      prfac * EXP(prexp * LOG(cloud_height))         ! in s-1 
    ENDIF
  ENDDO
ENDIF

!--------------------------
! Assign structure elements
!--------------------------

DO jj = 1, ics_len

  cell(jj) % centroid_x = cwi_clusters(jj) % cent_pos_x 
  cell(jj) % centroid_y = cwi_clusters(jj) % cent_pos_y
  cell(jj) % flash_rate = lf_pr92_cwi(jj) 

ENDDO

!--------------------------------------------------------
! Distribute flashes around the centroid (space and time)
!--------------------------------------------------------

accum_flashes = 0.0_iintegers

DO jj = 1, ics_len
  
  accum_flashes = accum_flashes + &
  NINT(lf_pr92_cwi(jj) * 3600.0_ireals * hinclight)

ENDDO

IF (accum_flashes > 0) THEN
  ALLOCATE(fl_lon (accum_flashes), STAT = istat )
  ALLOCATE(fl_lat (accum_flashes), STAT = istat )
  ALLOCATE(fl_time(accum_flashes), STAT = istat )
ELSEIF (accum_flashes == 0) THEN
  ALLOCATE(fl_lon (2), STAT = istat )
  ALLOCATE(fl_lat (2), STAT = istat )
  ALLOCATE(fl_time(2), STAT = istat )
ENDIF

CALL distribute_flashes (cell, accum_flashes, fl_lon, &
                         fl_lat, fl_time, cwi_lab)

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(cwi_clusters, STAT = istat)
DEALLOCATE(lf_pr92_cwi,  STAT = istat)
DEALLOCATE(cell,         STAT = istat)
DEALLOCATE(fl_lon,       STAT = istat)
DEALLOCATE(fl_lat,       STAT = istat)
DEALLOCATE(fl_time,      STAT = istat)

!============================================================================
! End subroutine pr92_cwi
!============================================================================

END SUBROUTINE pr92_cwi

!============================================================================
! Module procedure YMUK_2009
!============================================================================

SUBROUTINE ymuk_2009

!-----------------------------------------------------------------------------
! Description:
!   This subroutine calculated the flash rate based on the simple
!   relationship put forth by Yoshida et al. (2009, JGR).
!
! Method:
!  As in the PR parameterizations, but with minor pre-processing of the
!  upward-motion field to account for the cold cloud depth
!-----------------------------------------------------------------------------

IMPLICIT NONE

!-----------------------------------------------------------------------------
! Subroutine arguments
!-----------------------------------------------------------------------------

! None

!-----------------------------------------------------------------------------
! Local variables
!-----------------------------------------------------------------------------

! Local parameters
! ----------------

REAL (KIND=ireals), PARAMETER ::  &
  alpha    = 4.9_ireals,          &
  beta     = -6.1_ireals,         &
  w_binmar = 2.0

! Local scalars:
! -------------

INTEGER (KIND=iintegers) ::  &
  i, j, k,           &  
  ii, jj, kk,        & ! More loop indices (1D loops)
  i_glob,            & ! From parallel utilities 
  j_glob,            &
  i_loc ,            &
  j_loc ,            &
  isubdomain,        &
  ierror, impierror, & ! Error flags 
  istat,             & ! allocation error flag  
  nt,                & ! Current time step  
  lun,               & ! Unit number for IO 
  ics_len,           & ! length of csize (unique entries)
  count 

INTEGER (KIND=iintegers) ::  &
  accum_flashes        ! total number of flashes in hinclight period

REAL (KIND=ireals)       ::  &
  storm_height,      & ! updraft top                 [m]
  depth                ! cold cloud depth            [km]

! 3D arrays
!----------

 REAL    (KIND=ireals   ) ::  &
   w_hfl(ie, je, ke)             ! de-staggered UVVs 

INTEGER (KIND=iintegers) ::  &
  w_bin(ie, je, ke),         &
  w_lab(ie, je, ke),         &
  q_bin(ie, je, ke),         &  ! Filtered field
  q_lab(ie, je, ke)             ! Labeled (binary) field

! 2D arrays
! ---------

! None

! 1D arrays
! ---------

INTEGER (KIND=iintegers) ::  &
  icsize(csize_length)

! Logical variables
! -----------------

LOGICAL :: &
  lverbose = .TRUE.              ! Print flash rate, etc. 

! Allocatable structures and arrays
!----------------------------------

REAL (KIND=ireals), ALLOCATABLE ::  &
  fl_lon (:),                       &
  fl_lat (:),                       &
  fl_time(:)                          

TYPE(cluster_info), ALLOCATABLE ::  &
  w_clusters(:)

TYPE(storm), ALLOCATABLE        ::  &
  cell(:)

REAL (KIND=ireals), ALLOCATABLE ::  &
  lf_ymuk_2009(:)

!============================================================================
! Start procedure
!============================================================================

!------------------------------------------------------------------------------
! Section 1.1:
!   Prepare vertical-velocity field for label procedure
!------------------------------------------------------------------------------

! Destagger w-field

w_hfl (:,:,1:ke) = 0.5 * (w(:,:,1:ke,nnow) + w(:,:,2:ke+1,nnow))

! Only consider "cold-cloud depth"

WHERE (t(:,:,:,nnow) >= 273.15_ireals) w_hfl(:,:,:) = 0.0_ireals

w_bin = bin_field(w_hfl, w_binmar)

!------------------------------------------------------------------------------
! Section 2.1:
!   Call labeling procedure and allocate arrays with cs_count
!   Calculate flash rate
!------------------------------------------------------------------------------

CALL label (w_bin, w_lab, icsize) 

ics_len = cs_count

ALLOCATE(w_clusters  (ics_len), STAT = istat)
ALLOCATE(lf_ymuk_2009(ics_len), STAT = istat)
ALLOCATE(cell        (ics_len), STAT = istat)

! Initialize arrays/objects
! -------------------------

cell % flash_rate = 0.0_ireals
cell % centroid_x = 0_iintegers
cell % centroid_y = 0_iintegers
zrs               = 0_iintegers

DO ii = 1, ics_len
  zrs(ii) = 22_iintegers  ! Corresponds to breakdown-altitude of more 
ENDDO                     ! sophisticated scheme

!-----------------------------------------------------------------------------
! Calculate flash rate based on Yoshida et al., 2009; if cells are too small
! (less than 20 gridpoints, flash frequency is set to zero 
!-----------------------------------------------------------------------------

lf_ymuk_2009 = 0.0_ireals

IF (ics_len > 0) THEN
  CALL cluster_analysis (w_clusters, w_lab, icsize) 

  DO jj = 1, ics_len
    IF (w_clusters(jj) % pixels > 10) THEN   ! Reduce gravity-wave/Alps effect
     
      depth = 1.E-12_ireals + 1.0E-3_ireals * &                         ! km
      ABS(w_clusters(jj) % top_height - w_clusters(jj) % bottom_height) 
 
      lf_ymuk_2009(jj) = &
      EXP(beta * LOG(10.0_ireals)) * EXP(alpha * LOG(depth))  ! in s-1

      IF (lverbose .AND. my_cart_id == 0) THEN
        print *, '****************************************************'
        print *, '   SRC_LIGHTNING.YMUK_2009: Verbose output'
        print *, '****************************************************'
        print *, ''
        print '(a35, i7, a25, i7, f7.2)', 'Max number of cells:',          &
                  ics_len, 'at time step/hour:',                    &
                  ntstep, ntstep * dt / 3600.0_ireals
        print '(a35, i5)', 'Details of storm (label):', jj
        print *, ''
        print '(a35, i6, a1, i4)', 'Position (X,Y):',    &
               w_clusters(jj) % cent_pos_x, ',',  w_clusters(jj) % cent_pos_x
        print '(a35, f8.1, a10)', 'Top height:',    &
        1.0E-3_ireals * w_clusters(jj) % top_height  , ' km'
        print '(a35, f8.1, a10)', 'Cold cloud depth:', depth, ' km'   
        print '(a35, f8.4, a10)', 'Flash rate:',    &
        60.0_ireals * lf_ymuk_2009(jj),          '1/min'
        print *, ''
      ENDIF
    ENDIF  ! > 10 pixels  
  ENDDO 
ENDIF  ! icslen

!--------------------------
! Assign structure elements
!--------------------------

DO jj = 1, ics_len

  cell(jj) % centroid_x = w_clusters(jj) % cent_pos_x 
  cell(jj) % centroid_y = w_clusters(jj) % cent_pos_y
  cell(jj) % flash_rate = lf_ymuk_2009(jj) 

ENDDO

!--------------------------------------------------------
! Distribute flashes around the centroid (space and time)
!--------------------------------------------------------

accum_flashes = 0.0_iintegers

DO jj = 1, ics_len
  accum_flashes = accum_flashes + &
  NINT(lf_ymuk_2009(jj) * 3600.0_ireals * hinclight)
ENDDO

IF (accum_flashes > 0) THEN
  ALLOCATE(fl_lon (accum_flashes), STAT = istat )
  ALLOCATE(fl_lat (accum_flashes), STAT = istat )
  ALLOCATE(fl_time(accum_flashes), STAT = istat )
ELSEIF (accum_flashes == 0) THEN
  ALLOCATE(fl_lon (2), STAT = istat )
  ALLOCATE(fl_lat (2), STAT = istat )
  ALLOCATE(fl_time(2), STAT = istat )
ENDIF

CALL distribute_flashes (cell, accum_flashes, fl_lon, fl_lat, fl_time, w_lab)

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(w_clusters,      STAT = istat)
DEALLOCATE(lf_ymuk_2009,    STAT = istat)
DEALLOCATE(cell,            STAT = istat)
DEALLOCATE(fl_lon,          STAT = istat)
DEALLOCATE(fl_lat,          STAT = istat)
DEALLOCATE(fl_time,         STAT = istat)

!============================================================================
! End subroutine YMUK_2010
!============================================================================

END SUBROUTINE ymuk_2009

!============================================================================
! Module procedure GETAL_2001
!============================================================================

SUBROUTINE getal_2001

!-------------------------------------------------------------------------------
! Description:
!  This module procedure calculates the flash frequency based on the parameter-
!  ization by Grewe et al. (2001; Atmos. Environment, 35, 3421-3433). 
!
! Method:
!  Hoshen and Copleman's cluster identification algorithm, performed by the
!  subroutine LABEL.  Based on this information, applied to the vertical-
!  velocity field, the mean vertical velocity per cell as well as its depth
!  can be calculated and the flash frequency be deduced. 
!------------------------------------------------------------------------------

IMPLICIT NONE

! Local scalars:
! -------------
INTEGER (KIND=iintegers) ::  &
  k     ,            & ! loop index in vertical direction
  i     ,            & ! loop index in x-direction
  j     ,            & ! loop index in y-direction
  i_glob,            & ! From parallel utilities 
  j_glob,            &
  i_loc ,            &
  j_loc ,            &
  isubdomain,        &
  ierror, impierror, & ! Error flags 
  istat,             & ! allocation error flag  
  nt,                & ! Current time step  
  lun,               & ! Unit number for IO 
  ics_len,           & ! length of csize (unique entries)
  count, jj

INTEGER (KIND=iintegers) ::  &
  accum_flashes        ! total number of flashes in hinclight period

REAL (KIND=ireals)          ::  &
  storm_height,      &  ! updraft top                 [m]
  w_mean,            &  ! mean updraft velocity per region
  depth                 ! depth of w-region

! Local parameters
!-----------------

REAL (KIND=ireals), PARAMETER ::  &
  grfac = 8.35490e-09_ireals,   &  ! 5.01294e-07 / 60.0; factor in
                                   ! GR01 formula
  grexp = 2.64260_ireals           ! exponent in this equation

REAL (KIND=ireals), PARAMETER ::  &
  w_binmar = 2.0

! Local arrays
!-------------

REAL    (KIND=ireals   ) ::  &  
  w_hfl(ie, je, ke)             ! de-staggered UVVs 

INTEGER (KIND=iintegers) ::  &
  w_bin(ie, je, ke),         &
  w_lab(ie, je, ke),         &
  q_bin(ie, je, ke),         &  ! Filtered field
  q_lab(ie, je, ke)             ! Labeled (binary) field

INTEGER (KIND=iintegers) ::  &
  icsize(1000000_iintegers)

! Allocatable arrays and derived types
!-------------------------------------

REAL (KIND=ireals), ALLOCATABLE ::  &
  fl_lon (:),                       &
  fl_lat (:),                       &
  fl_time(:)                          

TYPE(cluster_info), ALLOCATABLE ::  &
  w_clusters(:)

TYPE(storm), ALLOCATABLE        ::  &
  cell(:)

REAL (KIND=ireals), ALLOCATABLE ::  &
  lf_gr01(:)

! Logical variables
! -----------------

LOGICAL                          :: &
  lverbose = .TRUE.

! String variables
! ----------------

CHARACTER (LEN=4)  ::  &
  stats_spec

!=============================================================================
! Start program
!=============================================================================

!------------------------------------------------------------------------------
! Section 1.1:
!   Prepare vertical-velocity field for label procedure
!------------------------------------------------------------------------------

! Destagger w-field

w_hfl (:,:,1:ke) = 0.5 * (w(:,:,1:ke,nnow) + w(:,:,2:ke+1,nnow))

w_bin = bin_field(w_hfl, w_binmar)

!------------------------------------------------------------------------------
! Section 2.1:
!   Call labeling procedure and allocate arrays with cs_count
!   Calculate flash rate
!------------------------------------------------------------------------------

CALL label (w_bin, w_lab, icsize ) 

ics_len = cs_count

ALLOCATE(w_clusters(ics_len),      STAT = istat)
ALLOCATE(lf_gr01(ics_len),         STAT = istat)
ALLOCATE(cell(ics_len),            STAT = istat)

! Initialize arrays/objects
! -------------------------

cell % flash_rate = 0.0_ireals
cell % centroid_x = 0_iintegers
cell % centroid_y = 0_iintegers

zrs               = 0_iintegers

DO jj = 1, ics_len
  zrs(jj) = 22_iintegers  ! Corresponds to breakdown-altitude of more 
ENDDO                     ! sophisticated scheme

!-----------------------------------------------------------------------------
! Calculate flash rate based on Grewe et al, 2001; if cells are too small 
! (less than 10 gridpoints) flash frequency is set to zero 
!-----------------------------------------------------------------------------

IF (ics_len > 0) THEN
  CALL cluster_analysis (w_clusters, w_lab, icsize, w_hfl)

  DO jj = 1, ics_len
    IF (w_clusters(jj) % pixels > 10) THEN   ! Reduce gravity-wave/Alps effect

      depth  = ABS(w_clusters(jj) % top_height - &
                   w_clusters(jj) % bottom_height)
      w_mean =     w_clusters(jj) % average

      ! to stay out of trouble with LOG-function 

      IF ( (SQRT(depth) * w_mean) > 0.0 ) THEN 

        lf_gr01(jj) = &
        grfac * EXP(grexp * LOG(SQRT(depth) * w_mean))     ! in 1/s !

        IF (lverbose .AND. my_cart_id == 0) THEN
          print *, '****************************************************'
          print *, '   SRC_LIGHTNING.GETAL_2001: Verbose output'
          print *, '****************************************************'
          print *, ''
          print '(a35, i7, a25, i7, f7.2)', 'Max number of cells:',          &
                     ics_len, 'at time step/hour:',                    &
                      ntstep, ntstep * dt / 3600.0_ireals
          print '(a35, i5)', 'Details of storm (label):', jj
          print *, ''
          print '(a35, i6, a1, i4)', 'Position (X,Y):',    &
                 w_clusters(jj) % cent_pos_x, ',',  w_clusters(jj) % cent_pos_x
          print '(a35, f9.1, a12)', 'Top height:',    &
           1.0E-3_ireals * w_clusters(jj) % top_height  , ' km'
          print '(a35, f9.1, a12)', 'Mean updraft:',  w_mean, 'm/s'
          print '(a35, f9.1, a12)', 'Depth:',  depth, ' km'
          print '(a35, f9.1, a12)', 'w * SQRT(D):', SQRT(depth) * w_mean, 'm**(3/2)/s'
          print '(a35, f9.4, a12)', 'Flash rate:',    &
          60.0_ireals * lf_gr01(jj),          '1/min'
          print *, '' 
        ENDIF
      ELSE
        lf_gr01(jj) = 0.0_ireals
      ENDIF
    ELSE IF (w_clusters(jj) % pixels <= 10) THEN
      lf_gr01(jj) = 0.0_ireals
    ENDIF
  ENDDO
ELSE
  DO jj = 1, ics_len
    lf_gr01(jj) = 0.0_ireals
  ENDDO
ENDIF

!--------------------------
! Assign structure elements
!--------------------------

DO jj = 1, ics_len

  cell(jj) % centroid_x = w_clusters(jj) % cent_pos_x 
  cell(jj) % centroid_y = w_clusters(jj) % cent_pos_y
  cell(jj) % flash_rate = lf_gr01(jj) 

ENDDO

!--------------------------------------------------------
! Distribute flashes around the centroid (space and time)
!--------------------------------------------------------

accum_flashes = 0.0_iintegers

DO jj = 1, ics_len
  
  accum_flashes = accum_flashes + &
  NINT(lf_gr01(jj) * 3600.0_ireals * hinclight)

ENDDO

IF (accum_flashes > 0) THEN
  ALLOCATE(fl_lon (accum_flashes), STAT = istat )
  ALLOCATE(fl_lat (accum_flashes), STAT = istat )
  ALLOCATE(fl_time(accum_flashes), STAT = istat )
ELSEIF (accum_flashes == 0) THEN
  ALLOCATE(fl_lon (2), STAT = istat )
  ALLOCATE(fl_lat (2), STAT = istat )
  ALLOCATE(fl_time(2), STAT = istat )
ENDIF

CALL distribute_flashes (cell, accum_flashes, fl_lon, fl_lat, fl_time, w_lab)

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(w_clusters,      STAT = istat)
DEALLOCATE(lf_gr01,         STAT = istat)
DEALLOCATE(cell,            STAT = istat)

DEALLOCATE(fl_lon,          STAT = istat)
DEALLOCATE(fl_lat,          STAT = istat)
DEALLOCATE(fl_time,         STAT = istat)

!============================================================================
! End subroutine GETAL_2001
!============================================================================

END SUBROUTINE getal_2001

!============================================================================
! Module procedure BIN_FIELD
!============================================================================

FUNCTION bin_field (field, binmar)      &
RESULT(bin_field_result) 

!-----------------------------------------------------------------------------
! Description:
!
!  This internal procedure transforms the input field into a binary one, which
!  may subsequently be labeled.  The field contains 0's and -1's, based on
!  bnmar which determines, beyond which values the field is filtered. 
!
! INPUT ARGUMENTS
! ----------------
!
!   field :   The field which is "binarized" and labeled
!
!   binmar:   Binarization margin: A filter, beyond which the values are
!             set to zero, or -1, respectively.  This is required because
!             the field which is to be labeled may only contain -1 and zeros.
!
! OUTPUT
! ------
!
!   A field that contains consecutive numbers as unique labels where the 
!   input field contained isolated clusters/blobs of -1's.
!
!   TO-DO:
!   ------
!
! * Build in switch whether filter should single out values LT or GT binmar
!   use "OPTIONAL" attribute (Check with PRESENT). 
!   Also LT or LE (LT, LE) switches are needed
!-----------------------------------------------------------------------------

IMPLICIT NONE
 
! Function input arguments
!--------------------------

REAL (KIND = ireals) ::        &
  binmar                         ! binarization margin
   
REAL (KIND = ireals) ::        &
  field           (ie, je, ke)   ! input field

! Function result
! ---------------

INTEGER (KIND=iintegers) ::    &
  bin_field_result(ie, je, ke)   ! output field
 
! Local parameters
!-----------------

! Local scalars:
! -------------
  INTEGER (KIND=iintegers) ::  &
    k,                         & ! loop index in vertical direction
    i,                         & ! loop index in x-direction
    j                            ! loop index in y-direction

! Local arrays
!-------------

 REAL  (KIND = ireals) ::      &
  ut_field        (ie, je, ke)

!--------------------------------------------------------------------------

! Renaming local variable because otherwise the input field would be changed
! (dummy argument and function argument are pointing to the 
! same memory location)

ut_field = field

! Make binary field (0, -1) of the original field based on the margin
! "binmar".

WHERE (ut_field <= binmar) ut_field = 0
WHERE (ut_field /= 0     ) ut_field = -1

bin_field_result = INT(ut_field)

!----------------------------------------------------------------------------
! End function BIN_FIELD
!----------------------------------------------------------------------------

END FUNCTION bin_field

!============================================================================
! Subroutine LABEL
!============================================================================

!----------------------------------------------------------------------------
!
! Description:
!
! Identify and label contiguous regions of properties defined in bin_field
! function.  The algorithm is based on Hoshen and Kopelman (1976).  This
! procedure accepts input of an integer array containing 0's and -1's,
! where the clusters of adjacent -1's are labeled.  The NEWS-neighborhood
! rule is used to determine pixel adjacency.  Parallelization has been
! realized as in Constantin et al. (supercomp. appl., 1997)
!
! Method:
!
! This program identifies and labels clusters using the NEWS neighborhood
! rule with the Hoshen-Kopelman algorithm.  Its fast processing time is
! owed to the maintainence of a separate array, CSIZE, which contains
! information about the number of pixels in one cluster, as well as cluster
! coalescence.  Initially, the pixels are given temporary labels, which
! are adjusted according to CSIZE information in a second pass through
! the array.
!
! The array is traversed row by row, and previously-labeled pixels are
! checked every time an occupied site is encountered.
!
! CSIZE is a 1D array, initialized to contain all clusters (checkerboard
! distribution).  Every cluster label is used as index to a CSIZE element.
! This element either contains the number of pixels in the cluster, or
! a negative number.  The absolute value of this number is the index
! to the cluster which the current cluster belongs to.  It may point
! to another CSIZE element which is negative, but eventually a positive
! number is reached, representing the number of elements in this cluster
! (and of all the clusters pointing to this cluster).
! The pointer path is found iteratively by the function PROPLAB (for
! "proper label").
! CSIZE is updated every time an occupied site is encountered.
!
! In the parallel implementation, CSIZE is declared as array that
! has 1E6 elements.
! CSIZE is defined for all processes such as to be able to contain all
! clusters of the TOTAL domain;  however, each process uses only a 
! well-defined part of CSIZE.
!----------------------------------------------------------------------------

SUBROUTINE label (field_in, field_out, csize_consec) 

IMPLICIT NONE

!----------------------------------------------------------------------------
! Subroutine arguments
!----------------------------------------------------------------------------

INTEGER (KIND=iintegers), INTENT(IN)  ::   &
  field_in    (ie,je,ke)

INTEGER (KIND=iintegers), INTENT(OUT) ::   &
  field_out   (ie,je,ke)                  

INTEGER (KIND=iintegers), INTENT(OUT) ::   & 
  csize_consec(csize_length)  

! Local constants and parameters; utility variables
!---------------------------------------------------

INTEGER (KIND=iintegers), PARAMETER   :: &
  dcount = 1

INTEGER (KIND=iintegers)  ::  &
  iejeh,                      & 
  i, j, k,                    &
  ii, jj,                     &         !  Loop variables
  incr

CHARACTER (LEN = 50)      ::  &
   header

CHARACTER (LEN=25) ::         &
    yerrmsg                           ! for MPI error message

! Local scalar variables
!-----------------------

INTEGER (KIND = iintegers)   ::  &
  lstat                               ! status of allocation

INTEGER (KIND=iintegers)     ::  &
  cs_size, num_clust,            &    ! size of csize;
  proper,                        &    ! number of clusters
  lun                                 ! unit number for write

INTEGER (KIND=iintegers)     ::  &
  count, ccount, counter              ! COUNT function output,
                                      ! counter variable

! Naming convention: "nb" stands for neighbor; t for target (of pointer path)

INTEGER (KIND=iintegers) ::        &
  s, r,                            &
  min_n, max_n,                    &
  target_min, target_max,          &
  target,                          &
  target_n1, target_n2, target_n3, &  ! targets of neighbors
  n1, n2, n3,                      &  ! neighbors
  position, eval, oval,            &  ! output from
                                      ! two_neighbors
  target_eval, target_oval,        &  ! their targets
  min_2nb, max_2nb,                &  ! min, max neighbors
  mint2nb, maxt2nb,                &  ! their targets
  positiont,                       &  ! output from two_n.
  evalt, ovalt,                    &  !     -- " --
  min_neighbor,                    &  ! min neighbor
  min3_index,                      &  ! index of -"-
  onb1, onb2,                      &  ! non-minumum neighb.
  mint3nb, onb1tar, onb2tar,       &  ! targets
  onbt,                            &  ! non-unique neighbor
  position2nb, eval2nb, oval2nb,   &  ! output two_elements
  target1nb,                       &  ! one neighb. target
  nb2eq, target_nb2eq,             &  ! neighbor if both
                                      ! identical; its target
  min_2nbt, max_2nbt,              &  ! min, 2 neighbors; targ.
  ierror, impierror, izerror          ! general and MPI error code 

INTEGER (KIND=iintegers) ::        &
  s_ovlap   , n_ovlap   ,          &  ! Overlapping clusters (north and south)
  indn      , inds      ,          &  ! indices for merge_array
  nedgeproc ,                      &  ! northernmost processor/subdomain 
  i_loc     , j_loc     ,          &
  loclen    ,                      &  ! "local" length of csize
  s_ovlap_tar, n_ovlap_tar,        &  ! Targets for global csize merging
  count_fe  , count_se  ,          &  ! Ordered-pair elements of merge_array
  merlen    ,                      &  ! length of merge_array 
  maxol     ,                      &  ! Maximum of overlapping clusters
                                      ! per row
  ope1    , ope2,                  &  ! ordered-pair elements 1 and 2
  count_feu, count_seu,            &  !
  southern_target, &                  ! target of southern portions of  
                                      ! multiply intersected cluster

  target_north, target_south

INTEGER (KIND=iintegers) :: iglob, jglob, vals

! For debugging/checking
!-----------------------

INTEGER (KIND=iintegers) ::             &
  global_field(ie_tot, je_tot, ke_tot), &
  global(ie_tot, je_tot, ke_tot),       &
  incr1, rz

!----------------------------------------------------------------------------
! Local arrays
!----------------------------------------------------------------------------

! 1D-arrays
! ---------

INTEGER (KIND=iintegers)  :: &   
  nb_targets(3),             & 
  neighbors (3),             &
  kzdims   (24),             &            ! Vertical dimension of sendbuf variables
  op        (2)                           ! ordered pair  

! 2D-arrays
! ---------

REAL (KIND=ireals)                ::  &
  field_2d(ie,je)                         ! for test purposes

INTEGER (KIND=iintegers)          ::  &
  test_arr(ie,je)

! 3D-arrays
! ---------

REAL (KIND=ireals)                ::  &
  field_prel(ie,je,ke)                    ! prelim. field; field_out has 
                                          ! wrong data type (INT) for MPI
INTEGER  (KIND=iintegers)         ::  &
  matrix(ie+1,je+1,ke1),              &             
  arr_3d(ie,je,ke)

!----------------------------------------------------------------------------
! Dynamic arrays
!----------------------------------------------------------------------------

INTEGER (KIND=iintegers), ALLOCATABLE :: &
  csize     (:),                        &
  csize_glob(:),                        &          
  csize_work(:),                        &
  merge_array(:),                       &  ! Local merge_array        
  utility_ma (:),                       &  ! Utility for merge array
  merge_array_glob(:),                  &  ! Globally-reduced version
  db1(:), db2(:)                           ! debug

!----------------------------------------------------------------------------
! Logical variables
!----------------------------------------------------------------------------

LOGICAL                               ::   &
  test    = .FALSE.,                       &        
  test1   = .FALSE.,                       &
  printdb = .FALSE.,                        & ! if true, print cluster info 
  flag, flagt, flag2nb

! For debugging:
! -------------

INTEGER (KIND=iintegers) :: &
  dbvar_0, dbvar_1,         &
  cound_db

!----------------------------------------------------------------------------
! Allocate memory 
!----------------------------------------------------------------------------

iejeh  = csize_length 
maxol  = n_overlapping_elements
loclen = NINT(REAL(iejeh) / REAL(nproc))  
merlen = nproc * maxol

IF (my_cart_id == 0 .AND. INT(nproc) * INT(loclen) < INT(iejeh)) THEN
  print *, '***********************************************************'
  print *, &
  'SRC_LIGHTNING.LABEL WARNING:' 
  print *, &
  'Sum of local CSIZE segments bigger than entire CSIZE array!'
   print *, 'nproc, loclen, nproc * loclen, iejeh:', &
             nproc, loclen, nproc * loclen, iejeh
   print *, '***********************************************************'
ENDIF

ALLOCATE (csize           (iejeh) , STAT = lstat)
ALLOCATE (csize_glob      (iejeh) , STAT = lstat)  ! csize after global reduction
ALLOCATE (utility_ma      (merlen), STAT = lstat)
ALLOCATE (merge_array     (merlen), STAT = lstat)  ! Merge_Array
ALLOCATE (merge_array_glob(merlen), STAT = lstat)

!============================================================================
! Begin program
!============================================================================

!----------------------------------------------------------------------------
! Section 1:
!
! Assign working array "matrix" (m+1)x(n+1)x(k+1) matrix
! such that a semi-halo around the upper-left border as well as at the top
! of the cube, is created.
!----------------------------------------------------------------------------

matrix = 0_iintegers
DO k = 1, ke
  DO j = jstartpar, jendpar
    DO i = istartpar, iendpar  
      matrix(i+1,j+1,k+1) = field_in(i,j,k) 
    ENDDO
  ENDDO
ENDDO

!----------------------------------------------------------------------------
! Section 1.1:
!
! Define processor-dependent variables used for parallel computations
! the appropriate segmentation of CSIZE would imply that incr starts from
!----------------------------------------------------------------------------

incr = my_cart_id * loclen    
incr1 = incr 

csize = 0_iintegers

!----------------------------------------------------------------------------
! Section 2:
! Traverse field and label it.  No cluster-fragment linking is
! performed during the first pass and has to be done in a second pass with
! the aid of csize.
!----------------------------------------------------------------------------

DO k = 2, ke1                ! First the uppermost horizontal slice
  DO j =  2, je+1             ! row-wise traversing
    DO i = 2, ie+1

      IF (matrix(i,j,k) == -1) THEN

        ! Some self-explanatory utility variables

        n1        = matrix(i-1,j,k)         ! western neighbor
        n2        = matrix(i,j-1,k)         ! northern neighbor
        n3        = matrix(i,j,k-1)         ! upper neighbor
        neighbors = (/n1, n2, n3/)
        min_n     = MIN(n1, n2, n3)
        max_n     = MAX(n1, n2, n3)

        !--------------------------------------------------------------------
        ! Case 1: No neighbors.
        !         Simply assign new label given by the counter
        !--------------------------------------------------------------------

        IF (n1 == 0 .AND. n2 == 0 .AND. n3 == 0) THEN
          s = incr + dcount     
          
          matrix(i,j,k) = s		
          csize(s)      = 1            
          incr          = incr + dcount
 
        !--------------------------------------------------------------------
        ! Case 2: All three neighbors labeled previously.
        !         Altogether, 6 cases to be considered
        !--------------------------------------------------------------------

        ELSEIF (n1 > 0 .AND. n2 > 0 .AND. n3 > 0) THEN

          ! Some utility assignments

          matrix(i,j,k) = min_n              ! Assign temporary label

          target_n1     = PROPLAB(csize, n1)
          target_n2     = PROPLAB(csize, n2)
          target_n3     = PROPLAB(csize, n3)
          target_min    = PROPLAB(csize, min_n)
          nb_targets    = (/target_n1, target_n2, target_n3/)

          ! Attempt, to call later - otherwise it's called for every
          ! site, and that will slow down the program ...

          CALL TWO_ELEMENTS(neighbors, flag, position, eval, oval)
          CALL TWO_ELEMENTS(nb_targets, flagt, positiont, evalt, ovalt)

          !------------------------------------------------------------------
          ! Case 2.1: All neighbors are different
          !------------------------------------------------------------------

          IF (n1 /= n2 .AND. n2 /= n3 .AND. n1 /= n3) THEN

            !----------------------------------------------------------------
            ! Case 2.1.1: All targets equal
            !----------------------------------------------------------------

            IF (target_n1 == target_n2 .AND. target_n2 == target_n3) THEN
              csize(target_n1) = 1 + csize(target_n1)

            !----------------------------------------------------------------
            ! Case 2.1.2: Two of the three targets equal
            !----------------------------------------------------------------

            ELSEIF (flagt) THEN

            ! Find position of minimum

              min_neighbor = MIN(n1, n2, n3)

              ccount = 0
              DO ii = 1, 3
                ccount = ccount + 1
                IF (neighbors(ii) == min_neighbor) min3_index = ccount
              ENDDO

              ! Find position of other elements

              IF (min3_index == 1) THEN
                onb1 = 2                   ! other neighbor 1
                onb2 = 3                   ! other neighbor 2
              ELSEIF (min3_index == 2) THEN
                onb1 = 1
                onb2 = 3
              ELSEIF (min3_index == 3) THEN
                onb1 = 1
                onb2 = 2
              ENDIF

              ! target of minimum neighbor

              mint3nb = PROPLAB(csize, min_neighbor)

              ! targets of other beighbors

              onb1tar = PROPLAB(csize, neighbors(onb1))
              onb2tar = PROPLAB(csize, neighbors(onb2))

              IF (mint3nb == onb1tar .AND. onb1tar /= onb2tar) THEN
                csize(mint3nb) = 1 + csize(mint3nb) + csize(onb2tar)
                csize(onb2tar) = -mint3nb
              ELSEIF (mint3nb == onb2tar .AND. onb2tar /= onb1tar) THEN
                csize(mint3nb) = 1 + csize(mint3nb) + csize(onb1tar)
                csize(onb1tar) = -mint3nb
              ELSEIF (onb1tar == onb2tar .AND. onb1tar /= mint3nb) THEN
                csize(mint3nb) = 1 + csize(mint3nb) + csize(onb1tar)
                csize(onb1tar) = -mint3nb
                csize(onb2tar) = -mint3nb
              ENDIF

            !----------------------------------------------------------------
            ! Case 2.1.3: All three targets are different
            !----------------------------------------------------------------

            ELSEIF (target_n1 /= target_n2 .AND. target_n2 /= target_n3 &
              .AND. target_n1 /= target_n3) THEN

              ! Find position of minimum

              min_neighbor = MIN(n1, n2, n3)

              ccount = 0
              DO ii = 1, 3
                ccount = ccount + 1
                IF (neighbors(ii) == min_neighbor) min3_index = ccount
              ENDDO

              ! Find position of other elements

              IF (min3_index == 1) THEN
                onb1 = 2                   ! other neighbor 1
                onb2 = 3                   ! other neighbor 2
              ELSEIF (min3_index == 2) THEN
                onb1 = 1
                onb2 = 3
              ELSEIF (min3_index == 3) THEN
                onb1 = 1
                onb2 = 2
              ENDIF

              ! Do cluster-fragment linking

              mint3nb = PROPLAB(csize, min_neighbor)
              onb1tar = PROPLAB(csize, neighbors(onb1))
              onb2tar = PROPLAB(csize, neighbors(onb2))

              csize(mint3nb) = 1 + csize(mint3nb) + csize(onb1tar) + &
                                   csize(onb2tar)
              csize(onb1tar) = - mint3nb
              csize(onb2tar) = - mint3nb

            ENDIF       ! Check of number of different targets

          !------------------------------------------------------------------
          ! Case 2.2: Two of three neighbors are different
          !------------------------------------------------------------------

          ELSEIF (flag) THEN

            target_oval = PROPLAB(csize, oval)
            target_eval = PROPLAB(csize, eval)

            !----------------------------------------------------------------
            ! Case 2.2.1: Both targets are equal
            !----------------------------------------------------------------

            IF (target_oval == target_eval) THEN
               csize(target_oval) = 1 + csize(target_oval)

            !----------------------------------------------------------------
            ! Case 2.2.2: Both targets are different
            !----------------------------------------------------------------

            ELSEIF (target_oval /= target_eval) THEN

              min_2nbt = PROPLAB(csize, min_n)
              max_2nbt = PROPLAB(csize, max_n)

              csize(min_2nbt) = 1 + csize(min_2nbt) + csize(max_2nbt)
              csize(max_2nbt) = -min_2nbt

            ENDIF

          !----------------------------------------------------------------
          ! Case 2.3: All neighbors are equal (and hence, all targets)
          !----------------------------------------------------------------

          ELSEIF (n1 == n2 .AND. n2 == n3) THEN
            csize(target_n1) = 1 + csize(target_n1)

          ENDIF                ! How many neighbors of the three are equal

        !--------------------------------------------------------------------
        ! Case 3: Two neighbors out of three are occupied (and labeled)
        !
        ! Two cases are checked: Two neighbors having been labeled previously
        !                        and one neighbor having been labeled
        !                        previously.  The latter case implies that
        !                        there is only one neighbor.
        !--------------------------------------------------------------------

        ELSEIF ( n1 > 0 .AND. n2 > 0 .AND. n3 == 0 .OR.         &
                 n1 > 0 .AND. n3 > 0 .AND. n2 == 0 .OR.         &
                 n2 > 0 .AND. n3 > 0 .AND. n1 == 0 ) THEN

          !------------------------------------------------------------------
          ! Case 3.1: Both neighbors are different
          !------------------------------------------------------------------

          ! Call of two_elements to obtain logical flag:
          ! If .FALSE., all three values are different; since one of them
          ! is zero, it directs the program to the desired branch.

          CALL TWO_ELEMENTS(neighbors, flag2nb, position2nb, eval2nb, oval2nb)

          IF (.NOT. flag2nb) THEN

            max_2nb    = MAX(n1, n2, n3)

            ! Temporarily overwrite the zero to find non-trivial minimum

            WHERE (neighbors == 0) neighbors = 10E5

            n1 = neighbors(1)
            n2 = neighbors(2)
            n3 = neighbors(3)

            min_2nb = MIN(n1, n2, n3)

            matrix(i,j,k) = min_2nb

            ! Recover neighbors

            WHERE (neighbors == 10E5) neighbors = 0

            n1 = neighbors(1)
            n2 = neighbors(2)
            n3 = neighbors(3)

            target_min = PROPLAB(csize, min_2nb)
            target_max = PROPLAB(csize, max_2nb)

            !--------------------------------------------------------------
            ! Case 3.1.1: Both targets are different
            !--------------------------------------------------------------

            IF (target_max /= target_min) THEN
              csize(target_min) = 1 + csize(target_min) + csize(target_max)
              csize(target_max) = - target_min 

            !--------------------------------------------------------------
            ! Case 3.1.2: Both targets are equal
            !--------------------------------------------------------------

            ELSEIF (target_max == target_min) THEN
              csize(target_min) = 1 + csize(target_min)
            ENDIF

          !----------------------------------------------------------------
          ! Case 3.2: Both neighbors (and hence, targets) are identical
          !----------------------------------------------------------------

          ELSE            ! flag2nb true, i.e., both neighbors are identical
            nb2eq            = MAX(n1, n2, n3)
            matrix(i,j,k)    = nb2eq  ! output from two_neighbors is (0, n, n)

            target_nb2eq        = PROPLAB(csize, nb2eq)
            csize(target_nb2eq) = 1 + csize(target_nb2eq)
          ENDIF

        !--------------------------------------------------------------------
        ! Case 4: Only one neighbor previously labeled
        !--------------------------------------------------------------------

        ELSEIF (n1 == 0 .AND. n2 == 0 .AND. n3 > 0 .OR.          &
                n1 == 0 .AND. n3 == 0 .AND. n2 > 0 .OR.          &
                n2 == 0 .AND. n3 == 0 .AND. n1 > 0) THEN

          matrix(i,j,k) = max_n

          ! Find proper label

          target1nb = PROPLAB(csize, max_n)
          csize(target1nb) = 1 + csize(target1nb)

        ENDIF     ! Number of neighbor
      ENDIF       ! site occupied

    ENDDO            ! i-loop
  ENDDO
ENDDO

!----------------------------------------------------------------------------
! Section 3.1:  
!  Do the proper labeling. If CSIZE has no non-zero elements, no further
!  statistics is attempted and the number of clusters is set to zero.
!----------------------------------------------------------------------------

cs_size   = COUNT(csize /= 0)
num_clust = COUNT(csize > 0)

IF (cs_size == 0 .AND. printdb) &
  print *, 'Process ', my_cart_id, 'LABEL: NO CLUSTERS FOUND'

IF (cs_size > 0) THEN
  ALLOCATE(csize_work(cs_size) , STAT = lstat)
  csize_work = 0_iintegers
  csize_work = &
  csize(1+(my_cart_id*loclen) : my_cart_id*loclen+cs_size)

  ! Proper labeling

  DO i = 1+(my_cart_id*loclen), my_cart_id*loclen+cs_size   
    IF (csize(i) < 0) THEN
      proper = PROPLAB(csize, i)
      WHERE (matrix == i) matrix = proper
    ENDIF
  ENDDO

ENDIF  ! csize containing non-trivial elements

! Assign 3D field (ie,je,ke) with contents of matrix

field_prel(1:ie,1:je,1:ke) = REAL(matrix(2:ie+1,2:je+1,2:ke+1))

!---------------------------------------------------------------------------
! Section 4: Exchange boundaries and set up merge_array ordered-pair array
!            Update global CSIZE
!  
! Method: The upper and lower three rows are used as halos so that they are
!         occupied by the values of the respective neighbors.  Then, the 
!         row at jendpar+1 is compared with the row at j = jendpar.  Where
!         clusters overlap (i.e. "touch" each other), csize is updated.
!
!----------------------------------------------------------------------------

! Assign kzdims

kzdims(:) = 0_iintegers
kzdims(1) = ke 

! Exchange values; halo depth is 3 (nboundlines) grid points.

CALL exchg_boundaries                                                        &
  (nnow+39, sendbuf, isendbuflen, imp_reals, icomm_cart, ie, je,             &
  kzdims, jstartpar, jendpar, nboundlines, nboundlines, my_cart_neigh,       &
  21000+ntstep, .FALSE., ncomm_type, izerror, yerrmsg,                       &
  field_prel(:,:,:))

IF (izerror /= 0) THEN
  ierror = izerror
  yerrmsg = 'LABEL: MPI EXCHANGE ERROR' 
  RETURN
ENDIF

!----------------------------------------------------------------------------
! Section 4.1:
!
! * Spread local CSIZE contents to all processes via MPI global reduction
!   routine.
! 
! * Find processor that covers northern edge of domain to limit the loop to
!   those processors that have a northern neighbor.  
!
! * Scan through the edge rows, look for overlapping clusters, 
!   and set up merge_array (ordered neighbor pairs)
!----------------------------------------------------------------------------

! Distribute entire CSIZE array to all processes

csize_glob = 0_iintegers

CALL MPI_ALLREDUCE   &
 (csize, csize_glob, iejeh, imp_integers, MPI_SUM, icomm_cart, izerror)       

! Determine processor at northern boundary edge (nedgeproc)

CALL ij_local (ie_tot-3, je_tot-3, i_loc, j_loc, nedgeproc, ierror)

! Set up Merge_Array

merge_array = 0_iintegers
count_fe    = my_cart_id * maxol + 1 
count_se    = my_cart_id * maxol + 2

IF (my_cart_id /= nedgeproc) THEN
  DO  k = 1, ke
    DO i = 1, ie
      n_ovlap = INT(field_prel(i,jendpar+1,k))    
      s_ovlap = INT(field_prel(i,jendpar  ,k)) 
      IF ( s_ovlap  /= 0 .AND. n_ovlap /= 0 ) THEN
         merge_array(count_fe:count_se) = (/ n_ovlap, s_ovlap /)
         count_fe = count_fe + 2
         count_se = count_se + 2
      ENDIF
    ENDDO
  ENDDO
ENDIF

! Scan through merge_array in parallel and eliminate redundant entries
! within the local processor scopes 
  
utility_ma = 0_iintegers 
count_feu  = my_cart_id * maxol + 1
count_seu  = my_cart_id * maxol + 2

DO i = my_cart_id * maxol + 1, (my_cart_id+1) * maxol, 2 
  ope1 = merge_array(i)
  ope2 = merge_array(i+1)
  op = (/ ope1, ope2 /) 
  IF (ope1 /= 0 .AND. ope2 /= 0) THEN
    DO ii = my_cart_id * maxol + 1, (my_cart_id+1) * maxol, 2
      IF (merge_array(ii)   == ope1 .AND. &
          merge_array(ii+1) == ope2) THEN
        merge_array(ii)                  = 0_iintegers
        merge_array(ii+1)                = 0_iintegers
        utility_ma (count_feu:count_seu) = op
      ENDIF
    ENDDO 
    count_feu = count_feu + 2
    count_seu = count_seu + 2
  ENDIF
ENDDO

! Broadcast result to all processes

merge_array_glob = 0_iintegers

CALL MPI_ALLREDUCE                             &
   (utility_ma, merge_array_glob, merlen,      &
   imp_integers, MPI_SUM, icomm_cart, izerror)

counter = 0
DO ii = 1, merlen
  IF (merge_array_glob(ii) /= 0) THEN
    counter = counter + 1
    IF (printdb) print *, my_cart_id, 'global merge array: ',  &
                          counter, merge_array_glob(ii)
  ENDIF
ENDDO

!----------------------------------------------------------------------------
! Section 4.2: Update global csize
!  The usual cluster-fragment linking is performed, but the global matrix
!  (field_prelim) and merge_array fields are updated right away.  This
!  avoids PROPLAB-Calls and the problem of linking multiply connected
!  regions, some of which are properly labeled while others are not.
!----------------------------------------------------------------------------

DO i = 1, merlen-1, 2
  n_ovlap = merge_array_glob(i)
  s_ovlap = merge_array_glob(i+1)

  IF (n_ovlap /= 0 .AND. s_ovlap /= 0 .AND. s_ovlap == n_ovlap) THEN 
    IF (printdb) THEN
      print *, ''
      print *, 'Previous merging has occurred', &
      s_ovlap, n_ovlap
    ENDIF
  ELSEIF (n_ovlap /= 0 .AND. s_ovlap /= 0 .AND. n_ovlap /= s_ovlap) THEN

    IF (printdb) THEN
      print *, my_cart_id, 'MERGE_ARRAY ORIGINAL: ', merge_array_glob(i), merge_array_glob(i+1)
      print *, ''
      print *, my_cart_id, 'Overlaps', n_ovlap, s_ovlap 
    ENDIF

    target_north = n_ovlap 
    target_south = s_ovlap

    IF (printdb) &
      print *, my_cart_id, 'before: ', csize_glob(target_north), csize_glob(target_south)

    csize_glob(target_north) = csize_glob(target_north) + csize_glob(target_south)
    csize_glob(target_south) = -target_north

    IF (printdb) &
      print *, my_cart_id, 'after: ', csize_glob(target_north), csize_glob(target_south) 

    WHERE (field_prel == REAL(target_south)) field_prel = REAL(target_north)
    WHERE (merge_array_glob == target_south) merge_array_glob = target_north     

    IF (printdb) THEN
      print *, my_cart_id, 'Global field relabeling: ', target_south, ' becomes ', target_north
      print *, my_cart_id, 'MERGE_ARRAY MODIFIED: ', merge_array_glob(i), merge_array_glob(i+1)
      print *, ''
    ENDIF 

  ENDIF
ENDDO

!----------------------------------------------------------------------------
! Section 4.3: 
!  Two final passes for proper and consecutive labeling with the aid of the
!  global CSIZE array
!----------------------------------------------------------------------------

! Do consecutive labeling for entire domain ...
  
cs_count     = 0_iintegers
csize_consec = 0_iintegers

DO i = 1, iejeh
  IF (csize_glob(i) > 0) THEN
    cs_count = cs_count + 1
    csize_consec(cs_count) = csize_glob(i)
    WHERE (field_prel == REAL(i)) field_prel = REAL(cs_count)
  ENDIF
ENDDO

field_out = INT(field_prel)

!----------------------------------------------------------------------------
! Section 5: Some cluster statistics
!----------------------------------------------------------------------------

num_clust = 0_iintegers

IF (cs_size > 0) THEN

  IF (printdb) THEN
    print *, 'Local statistics for process number ', my_cart_id
    print *, ''
  ENDIF

  num_clust = COUNT(csize > 0)

  IF (printdb) THEN
    DO i = 1+my_cart_id * loclen, my_cart_id * loclen+cs_size
      print *, my_cart_id, i, csize(i), csize_glob(i)
    ENDDO

    print *, ''
    print *, my_cart_id, ' Process  Cluster  number of elements'
    print *, '-------------------------------------------------'

    DO i = 1, cs_size  
      IF (csize_work(i) > 0) print '(3i8)', i, csize_work(i)
    ENDDO
    print *, my_cart_id,  'Number of clusters:', num_clust
    print *, ''
  ENDIF              ! verbose output
ENDIF              ! csize containing nontrivial elements  

IF (printdb) THEN
  print *, my_cart_id, 'Global statistics at timestep:', ntstep
  print *, ''

  print *, my_cart_id, ' Cluster       number of elements'
  print *,            '----------------------------------'

  DO i = 1, cs_count  
     print '(3i8)', my_cart_id, i, csize_consec(i)
  ENDDO
ENDIF

!----------------------------------------------------------------------------
! Section 6: Deallocate dynamic arrays
!----------------------------------------------------------------------------

IF (cs_size > 0) THEN
  DEALLOCATE (csize_work    , STAT = lstat)
ENDIF

DEALLOCATE (csize           , STAT = lstat)
DEALLOCATE (csize_glob      , STAT = lstat)
DEALLOCATE (utility_ma      , STAT = lstat)
DEALLOCATE (merge_array     , STAT = lstat)
DEALLOCATE (merge_array_glob, STAT = lstat)

!============================================================================
! End subroutine label
!============================================================================

END SUBROUTINE LABEL

!============================================================================

!----------------------------------------------------------------------------
!
! SUBROUTINE TWO_ELEMENTS:
!
! PURPOSE
!
! Internal procedure to determine, whether two of three elements
! contained in a three-element, 1D array, are identical.
! If so, a logical flag will be set to .TRUE.; in addition,
! the position of the value not equaling the two other elements will be
! determined, as well as the value of this element and the value of the
! two equal elements.  This should limit the number of IF-constructs in the
! calling subroutine.
!
! INPUT
!
! * elements
!  A three-element, 1D array
!
! OUTPUT
!
! * eflag:
!   A logical flag revealing whether or not precisely two elements are equal
!
!   --> Two of the three elements are equal:
!       eflag = .TRUE.
!   --> All elements being equal, or all elements being different:
!       eflag = .FALSE.
!
! * eposition:
!   The position of the element that is not equal than the others.  Set to
!   -999 if there is no uniquely different element (eflag == false).
!
! * evalue:
!   Value of the element being not equal to the two others. Set to
!   -999 if there is no uniquely different element (eflag == false).
!
! * ovalue:
!   Value of the other two elements.  Set to -999 if there is no uniquely
!   different element (eflag == false).
!
!----------------------------------------------------------------------------

SUBROUTINE TWO_ELEMENTS(elements, eflag, eposition, evalue, ovalue)

IMPLICIT NONE

!----------------------------------------------------------------------------
! Dummy arguments
!----------------------------------------------------------------------------

! Input arguments
! ---------------

INTEGER (KIND=iintegers), INTENT(IN) ::  &
  elements(3)

! Output arguments
! -----------------

INTEGER (KIND=iintegers), INTENT(OUT) :: &
  eposition,                             &
  evalue,                                &
  ovalue

LOGICAL :: &
  eflag

!INTEGER, DIMENSION(3), INTENT(IN)  :: elements
!LOGICAL, INTENT(OUT)               :: eflag
!INTEGER, INTENT(OUT)               :: eposition
!INTEGER, INTENT(OUT)               :: evalue
!INTEGER, INTENT(OUT)               :: ovalue

!----------------------------------------------------------------------------
! Local variables
!----------------------------------------------------------------------------

INTEGER       :: e1, e2, e3, emax, emin

!============================================================================
! Start subroutine two_elements
!============================================================================

e1 = elements(1)
e2 = elements(2)
e3 = elements(3)

! Abort if all equal or all different

IF ( (e1 == e2 .AND. e2 == e3) .OR. (e1 /= e2 .AND. &
      e2 /= e3 .AND. e1 /=e3) ) THEN

  eflag     = .FALSE.
  ovalue    = -999
  evalue    = -999
  eposition = -999

! Continue if only two of the three elements are equal
! Time-consuming IF-constructs are used in lack of a better idea ...

ELSEIF (e1 == e2 .AND. e2 /= e3 .OR.          &
        e1 == e3 .AND. e2 /= e3 .OR.          &
        e2 == e3 .AND. e1 /= e3) THEN

  eflag = .TRUE.

  ! Find which is the uniquely-valued element, as well as its position
  ! Not too elegant, and not debugged yet

  IF (e1 == e2) THEN
    eposition = 3
    evalue    = elements(3)
    ovalue    = elements(2)
    IF ( ovalue .NE. elements(1) ) print *, 'TWO_ELEMENTS ERROR - POINT 1'
  ELSEIF (e1 == e3) THEN
    eposition = 2
    evalue    = elements(2)
    ovalue    = elements(1)
    IF ( ovalue .NE. elements(3) ) print *, 'TWO_ELEMENTS ERROR - POINT 2'
  ELSEIF (e2 == e3) THEN
    eposition = 1
    evalue    = elements(1)
    ovalue    = elements(2)
    IF ( ovalue .NE. elements(3) ) print *, 'TWO_ELEMENTS ERROR - POINT 3'
  ENDIF

ENDIF

!============================================================================
! End subroutine two_elements
!============================================================================

END SUBROUTINE TWO_ELEMENTS

!============================================================================

!----------------------------------------------------------------------------
! Purpose and method:
!
! Internal function to iteratively find the end of a pointer path directed
! by the CSIZE array.
!
! For a given position, scan through csize and follow the pointer path,
! starting from the input index rr.  The path is followed until a positive
! csize element is found.  The index pointing to the first positive element
! that is encountered is output as result of the function.  The trivial
! case of the input index already pointing at the end of a pointer path
! is included.
!
! INPUT
!
!   n : csize array
!   rr: index to rr'th element of csize - start of the pointer path
!
! OUTPUT
!
!   prop = PROPLAB(csize, t):
!
!   Cluster label that the input index' cluster is associated with
!----------------------------------------------------------------------------

FUNCTION PROPLAB(n, rr) RESULT(prop)

IMPLICIT NONE

!----------------------------------------------------------------------------
! Local variables/dummy arguments
!----------------------------------------------------------------------------

INTEGER  (KIND=iintegers) :: &
  rr,                        &       ! input variables:
                                     !     rr: csize element
                                     !     (start of pointer path)
  prop,                      &       ! the end of the pointer path
  pp, ii,                    &       ! loop variables
  t1, abs_t1, ierr                   ! temporary/utility variables

INTEGER  (KIND=iintegers) :: &
  n(:)                               ! csize; assumed shape from 
                                     ! input

!----------------------------------------------------------------------------
! Do a simple iteration
!----------------------------------------------------------------------------

pp = 0
t1 = n(rr)

IF (n(rr) > 0 ) THEN
  prop = rr                                       ! Already at end of the path

ELSEIF (n(rr) < 0) THEN
  DO                                              ! open loop (to EXIT label)
    pp = 1 + pp
    abs_t1 = ABS(t1)
    IF (n(abs_t1) > 0) THEN
      prop = abs_t1
      EXIT
    ENDIF

    t1 = n(abs_t1)

    ! Terminate iteration after 1000 steps

    IF (pp .EQ. 1000) THEN

      ! Some debugging information ...

      print *, my_cart_id, &   
     'PROPLAB: END OF POINTER PATH NOT FOUND AFTER 1000 ITERATIONS ... ABORTING'
      print *, my_cart_id, 'Time step: ', ntstep
      print *, my_cart_id, ' CSIZE index, element: ', abs_t1, n(abs_t1)

      ! Terminate program

      CALL MPI_ABORT (MPI_COMM_WORLD, 100, ierr)
    ENDIF
  ENDDO
ELSEIF (n(rr) == 0) THEN  
! When csize element is zero (hinting at upstream error)
  print *, my_cart_id, &
  'PROPLAB ERROR: REFERENCED CSIZE ELEMENT IS ZERO ... ABORTING'
  print *, my_cart_id, 'Erroneous cluster label:', rr
  print *, my_cart_id, 'Time step: ', ntstep
  CALL MPI_ABORT (MPI_COMM_WORLD, 100, ierr)
ENDIF

!----------------------------------------------------------------------------
! End function PROPLAB
!----------------------------------------------------------------------------

END FUNCTION PROPLAB

!============================================================================
! Subroutine cluster_analysis
!============================================================================

!----------------------------------------------------------------------------
! Description:
!   This module procedure calculates the vertical and horizontal extent of
!   the different clusters, their volume, and their centroid positions.
!
! Method:
!   The centroid is simply the arithmetic mean of the positions of the
!   occupied gridopints.  The extents (and volume) are proportional to
!   the number of occupied gridpoints.
!
! Input:
!
!   * Properly and consecutively labeled field
!   * final (consecutive) CSIZE array
!   * Optionally, the field upon which BIN_FIELD function acted
!
! Output:
!   Structure containing arrays with cluster parameters
!
!----------------------------------------------------------------------------

SUBROUTINE cluster_analysis (cluster, labeled, lcsize, element_values) 

!----------------------------------------------------------------------------
! Subroutine arguments
!----------------------------------------------------------------------------

IMPLICIT NONE

INTEGER (KIND = iintegers), INTENT(IN)   ::   &
  labeled(ie,je,ke),              &   ! properly/consecutively labeled field
  lcsize(:)                           ! csize

INTEGER (KIND = iintegers)   ::   &
  lcs_len                            ! number of clusters

REAL (KIND=ireals), OPTIONAL, INTENT(IN)  ::  &
  element_values(ie,je,ke)

! Derived types
!--------------

TYPE(cluster_info), INTENT(OUT)    ::   &
  cluster(cs_count)

! Local parameters
!-----------------

! None

! Local scalars
!--------------

INTEGER (KIND = iintegers)   ::   &
  i, j, k, ii,                    &   ! loop variables
  iz1, iz2, iz3,                  &   ! loop variables
  label,                          &
  lstat, lerror,                  &
  top_x_loc, top_y_loc,           &   ! ij_local output  
  bottom_x_loc, bottom_y_loc,     &   ! ij_local output
  lprocess                            !     - " -  

! Local arrays
!-------------

INTEGER (KIND=iintegers)     ::    &
  cent_pos_x_arr(cs_count),        &   ! Containing global centroid x-component
  cent_pos_y_arr(cs_count),        &
  cent_pos_z_arr(cs_count),        &
  top_xx(cs_count),                &   ! x-position of uppermost pixel
  top_yy(cs_count),                &
  top_zz(cs_count),                &
  top_x_glob(cs_count),            &
  top_y_glob(cs_count),            & 
  top_z_glob(cs_count),            &   ! Globally reduced (MIN (= max height)) 
  bottom_xx(cs_count),             &   ! x-position of lowermost pixel
  bottom_yy(cs_count),             &   !
  bottom_zz(cs_count),             &   !
  bottom_x_glob(cs_count),         &   ! Globally-reduced
  bottom_y_glob(cs_count),         &
  bottom_z_glob(cs_count),         &
  iglobind(ie),                    &
  jglobind(je)

REAL (KIND=ireals)           ::    &
  cent_pos_x_util(cs_count),       & 
  cent_pos_y_util(cs_count),       &
  cent_pos_z_util(cs_count),       &
  cent_pos_x_arr_glob(cs_count),   &   ! Globally reduced (sum)
  cent_pos_y_arr_glob(cs_count),   &
  cent_pos_z_arr_glob(cs_count),   &
  height_top(cs_count),            &   ! Utility array (= top_height)
  height_top_glob(cs_count),       &
  height_bottom(cs_count),         &   ! Utility array (= bottom_height)
  height_bottom_glob(cs_count),    &
  cluster_sum_loc(cs_count),       &   ! Sum of all cluster elements
  cluster_sum_glob(cs_count)      

REAL (KIND=ireals)           ::    &
  hfl(ie,je,ke)                       ! Full levels
 
! For performance debugging

REAL (KIND=ireals) ::                            &
  ts_reduce_pos1, ts_reduce_pos2, ts_reduce_pos, &
  ts_bottom1, ts_bottom2, ts_bottom
 
! Dynamic arrays
!---------------

! None

LOGICAL ::  &
  verbose = .FALSE. 

!============================================================================
! Begin procedure
!============================================================================

!----------------------------------------------------------------------------
! Section 1.1: Find cluster centroids
!----------------------------------------------------------------------------

! Initializations (necessary as not all processors contain all clusters)

lcs_len = cs_count

hfl(:,:,1:ke) = 0.5 * (hhl(:,:,2:ke+1) + hhl(:,:,1:ke))

cent_pos_x_arr = 0_iintegers
cent_pos_y_arr = 0_iintegers
cent_pos_z_arr = 0_iintegers

top_xx = 0_iintegers
top_yy = 0_iintegers

jglobind = global_j

! Array elements are first determined by each processes separately
! and subsequently summed up in a global reduction operation.

DO ii = 1, lcs_len
  iz1 = 0_iintegers
  iz2 = 0_iintegers
  iz3 = 0_iintegers
  DO k = 1, ke
    DO j = jstartpar, jendpar
      DO i = istartpar, iendpar
        IF (labeled(i,j,k) == ii) THEN
          iz1 = iz1 + i
          iz2 = iz2 + jglobind(j)   
          iz3 = iz3 + k
        ENDIF
      ENDDO
    ENDDO
  ENDDO

  cent_pos_x_util(ii) =  REAL(iz1) / REAL(lcsize(ii))
  cent_pos_y_util(ii) =  REAL(iz2) / REAL(lcsize(ii))
  cent_pos_z_util(ii) =  REAL(iz3) / REAL(lcsize(ii))

ENDDO

! CALL MPI_BARRIER (icomm_cart, lerror)

ts_reduce_pos1 = MPI_WTIME()

CALL MPI_ALLREDUCE                            &
   (cent_pos_x_util, cent_pos_x_arr_glob, lcs_len, &
   imp_reals, MPI_SUM, icomm_cart, lerror)

CALL MPI_ALLREDUCE                            &
   (cent_pos_y_util, cent_pos_y_arr_glob, lcs_len, &
   imp_reals, MPI_SUM, icomm_cart, lerror)

CALL MPI_ALLREDUCE                            &
   (cent_pos_z_util, cent_pos_z_arr_glob, lcs_len, &
   imp_reals, MPI_SUM, icomm_cart, lerror)

ts_reduce_pos2 = MPI_WTIME()

ts_reduce_pos = ts_reduce_pos2 - ts_reduce_pos1

!----------------------------------------------------------------------------
! Section 1.2: Find bottom height of clusters
!----------------------------------------------------------------------------

ts_bottom1 = MPI_WTIME()

bottom_zz     = 0.0_ireals
bottom_x_glob = 0_iintegers
bottom_y_glob = 0_iintegers

DO ii = 1, lcs_len
  vertical_loop: DO k = ke, 1, -1    ! From bottom to top
    DO j = jstartpar, jendpar
      DO i = istartpar, iendpar
        IF (labeled(i,j,k) == ii) THEN
          bottom_zz(ii) = k
          EXIT vertical_loop 
        ENDIF 
      ENDDO
    ENDDO
  ENDDO vertical_loop
ENDDO

CALL MPI_ALLREDUCE (bottom_zz, bottom_z_glob, lcs_len, imp_integers,         &
                    MPI_MAX, icomm_cart, lerror)

! Find horizontal coordinates of cluster bottom

bottom_xx = 0.0_ireals
bottom_yy = 0.0_ireals

DO ii = 1, lcs_len
  k_loop: DO k = ke, 1, -1   ! From bottom to top
    DO j = jstartpar, jendpar
      DO i = istartpar, iendpar
        IF (labeled(i,j,k) == ii .AND. k == bottom_z_glob(ii)) THEN
          bottom_xx(ii) = i
          bottom_yy(ii) = global_j(j)
          EXIT k_loop
        ENDIF
      ENDDO
    ENDDO
  ENDDO k_loop
ENDDO 

! Let other processes know

CALL MPI_ALLREDUCE (bottom_xx, bottom_x_glob, lcs_len, imp_integers,         &
                        MPI_MAX, icomm_cart, lerror)

CALL MPI_ALLREDUCE (bottom_yy, bottom_y_glob, lcs_len, imp_integers,         &
                        MPI_MAX, icomm_cart, lerror)

ts_bottom2 = MPI_WTIME()
ts_bottom = ts_bottom2 - ts_bottom1

!----------------------------------------------------------------------------
! Section 1.2.1: Calculate the geometric heights of the clusters' bottoms
!----------------------------------------------------------------------------

height_bottom = 0_iintegers

DO ii = 1, lcs_len
  CALL ij_local                                    & 
     (bottom_x_glob(ii), bottom_y_glob(ii), bottom_x_loc,   &
      bottom_y_loc, lprocess, lerror)
  IF (my_cart_id == lprocess) THEN
    height_bottom(ii) = hfl(bottom_x_loc, bottom_y_loc, bottom_z_glob(ii))
  ENDIF
ENDDO

CALL MPI_ALLREDUCE (height_bottom, height_bottom_glob, lcs_len, imp_reals,   &
      MPI_MAX, icomm_cart, lerror)

!----------------------------------------------------------------------------
! Section 1.3: Find highest point occupied by individual clusters
!----------------------------------------------------------------------------

! Initialize top_z with a large number (GT ke)

top_zz     = 300_iintegers
top_x_glob = 0_iintegers
top_y_glob = 0_iintegers

DO ii = 1, lcs_len
  k_loop2: DO k = 1, ke
    DO j = jstartpar, jendpar
      DO i = istartpar, iendpar
        IF (labeled(i,j,k) == ii) THEN
           top_zz(ii) = k
           EXIT k_loop2
        ENDIF
      ENDDO
    ENDDO
  ENDDO k_loop2
ENDDO

CALL MPI_ALLREDUCE (top_zz, top_z_glob, lcs_len, imp_integers,       &
                    MPI_MIN, icomm_cart, lerror)                   

! Find horizontal coordinates of cluster top

DO ii = 1, lcs_len
  k_loop3:  DO k = 1, ke
    DO j = jstartpar, jendpar
      DO i = istartpar, iendpar
        IF (labeled(i,j,k) == ii .AND. k == top_z_glob(ii)) THEN
          top_xx(ii) = i
          top_yy(ii) = global_j(j)
          EXIT k_loop3
        ENDIF
      ENDDO
    ENDDO
  ENDDO k_loop3
ENDDO

! Let other processes know

CALL MPI_ALLREDUCE (top_xx, top_x_glob, lcs_len, imp_integers,         &
                        MPI_MAX, icomm_cart, lerror)

CALL MPI_ALLREDUCE (top_yy, top_y_glob, lcs_len, imp_integers,         &
                        MPI_MAX, icomm_cart, lerror)

!----------------------------------------------------------------------------
! Section 1.3.1: Calculate the geometric heights of the clusters' tops
!----------------------------------------------------------------------------

height_top = 0_iintegers

DO ii = 1, lcs_len
  CALL ij_local                                    & 
     (top_x_glob(ii), top_y_glob(ii), top_x_loc,   &
      top_y_loc, lprocess, lerror)
  IF (my_cart_id == lprocess) THEN
    height_top(ii) = hfl(top_x_loc, top_y_loc, top_z_glob(ii))
  ENDIF
ENDDO

CALL MPI_ALLREDUCE (height_top, height_top_glob, lcs_len, imp_reals,     &
                    MPI_MAX, icomm_cart, lerror)

!----------------------------------------------------------------------------
! Section 1.4: Calculate average value of cluster property (like mean upward
!              velocity in w-cluster)
!----------------------------------------------------------------------------

cluster(ii) % average = 0.0_ireals

IF (itype_light == 5) THEN

  cluster_sum_loc = 0.0_ireals

  DO ii = 1, lcs_len
    DO k = 1, ke
      DO j =  jstartpar, jendpar
        DO i = istartpar, iendpar
          IF (labeled(i,j,k) == ii) THEN
            cluster_sum_loc(ii) = cluster_sum_loc(ii) + element_values(i,j,k) 
          ENDIF
        ENDDO
      ENDDO
    ENDDO
  ENDDO

  cluster_sum_glob = 0.0_ireals

  CALL MPI_ALLREDUCE (cluster_sum_loc, cluster_sum_glob, lcs_len, imp_reals,  &
                      MPI_SUM, icomm_cart, lerror)
                        
  DO ii = 1, lcs_len
    cluster(ii) % average = cluster_sum_glob(ii) / REAL(lcsize(ii))
  ENDDO

ENDIF

!----------------------------------------------------------------------------
! Section 2: Pass contents to cluster structure
!----------------------------------------------------------------------------

DO ii = 1, lcs_len

  cluster(ii) % label         = ii
  cluster(ii) % pixels        = lcsize(ii)
  cluster(ii) % cent_pos_x    = NINT(cent_pos_x_arr_glob(ii))
  cluster(ii) % cent_pos_y    = NINT(cent_pos_y_arr_glob(ii))
  cluster(ii) % cent_pos_z    = NINT(cent_pos_z_arr_glob(ii))
  cluster(ii) % top_x         = top_x_glob(ii)
  cluster(ii) % top_y         = top_y_glob(ii)
  cluster(ii) % top_z         = top_z_glob(ii)
  cluster(ii) % bottom_height = height_bottom_glob(ii)
  cluster(ii) % top_height    = height_top_glob(ii)
 
ENDDO

IF (verbose) THEN
  print *, my_cart_id, 'Time step is: ', ntstep
  DO ii = 1, lcs_len
    print *, my_cart_id, 'label: ',    &
             cluster(ii) % label,      &
             cluster(ii) % pixels,     &
             cluster(ii) % cent_pos_x, &
             cluster(ii) % cent_pos_y, &
             cluster(ii) % cent_pos_z, & 
             cluster(ii) % top_x,      &
             cluster(ii) % top_y,      &
             cluster(ii) % top_z,      & 
             cluster(ii) % bottom_height, & 
             cluster(ii) % top_height  
  ENDDO
ENDIF

!============================================================================
! End module procedure cluster_analysis
!============================================================================

END SUBROUTINE cluster_analysis

!============================================================================
! Subroutine cluster_overlaps
!============================================================================

!----------------------------------------------------------------------------
! Description:
!   This internal subroutine looks for overlapping clusters of two classes.
!   The idea is to find the positively and negatively charged plates of
!   the capacitor.  The positive plate is assumed to be the anvil 
!   (or part of it), and the negative plate is the region where graupel
!   exists.    
!   Hence, regions where an ice cloud overlaps with a region of  
!   graupel are identified as potential thundercloud.  
!   The information about the top and bottom clusters are stored in the
!   derived-type variable ovl_clusters (for overlapping clusters).
!
! Method:
!   Calls of LABEL and CLUSTER_ANALYSIS routines separately for both cluster
!   types (e.g., MG and QI+QS).  Overlaps are assumed if at least half of
!   the lower area is covered by the cluster aloft.  This condition is checked 
!   by testing whether occupied pixels are found above the centroid of the
!   lower cluster.  If true, the information about the lower and upper 
!   clusters are stored in the ovl_clusters structure.
!
! Input:
! 
!   As input, this subroutine accepts the binary fields
!   
!   * top   : upper cluster
!
!   * bottom: lower cluster  
! 
!----------------------------------------------------------------------------

SUBROUTINE cluster_overlaps (top, bottom, bottom_lab)

!----------------------------------------------------------------------------
! Subroutine arguments
!----------------------------------------------------------------------------

IMPLICIT NONE

! Input arguments
! ---------------

INTEGER (KIND=iintegers), INTENT(IN) ::    &
  top         (ie,je,ke),                  &  ! binary field of upper cluster
  bottom      (ie,je,ke)                      ! binary field of lower cluster

! Output arguments
! ----------------

INTEGER (KIND=iintegers)  ::  &
  bottom_lab (ie,je,ke)                  ! Labeled field

!----------------------------------------------------------------------------
! Local variables
!----------------------------------------------------------------------------

! Derived data types

 TYPE(cluster_info), ALLOCATABLE  ::     &
   info_top   (:),                       &   
   info_bottom(:)

! 3D-arrays
! --------

REAL (KIND=ireals)  :: &
  rho_glob_temp (ie_tot, je_tot, ke_tot),    &
  rho_glob      (ie_tot, je_tot, ke_tot),    &
  rhfl_glob_red (ie_tot, je_tot, ke_tot),    &
  rhfl_glob     (ie_tot, je_tot, ke_tot),    &
  grar_glob_temp(ie_tot, je_tot, ke_tot),    &
  grar_glob     (ie_tot, je_tot, ke_tot),    &
  qg_glob_temp  (ie_tot, je_tot, ke_tot),    &
  qg_glob       (ie_tot, je_tot, ke_tot),    &
  mg_utility    (ie_tot, je_tot, ke_tot),    &
  rhfl          (ie,je,ke),                  &  ! full levels
  graupel_max_arr(ie_tot, je_tot, ke_tot)
 
INTEGER (KIND=iintegers)  ::                &
  top_lab             (ie,je,ke),           &
  top_lab_glob        (ie_tot, je_tot, ke), & ! non-reduced labeled upper field
  bottom_lab_glob     (ie_tot, je_tot, ke), & ! non-reduced labeled upper field
  bottom_lab_glob_red (ie_tot, je_tot, ke), & ! all-reduced labeled upper field
  top_lab_glob_red    (ie_tot, je_tot, ke)    ! all-reduced labeled upper field

! 2D-arrays
! ---------

! None

! 1D-arrays
! ---------

INTEGER (KIND=iintegers)   ::  &
  x_glob(ie), y_glob(je),      &          ! result of i/j_glob function
  tcsize(csize_length)  ,      &
  bcsize(csize_length)

REAL (KIND=ireals)  ::  &
  grau_ut(ke),          &                 ! Vertical 1D slice thru QG centroid
  riming_ut(ke)                           ! Vertical 1D slice thru GRAR centroid

INTEGER (KIND=iintegers), ALLOCATABLE :: &
  bottom_centroid_x(:),                  &
  bottom_centroid_y(:),                  &
  bottom_centroid_z(:),                  &   
  overlap_top_ut(:),                     &  ! Contains overlapping top
  overlap_bot_ut(:)                         ! Contains overlapping bottom

! Local scalars
! -------------

INTEGER (KIND=iintegers)   ::   &
  istat, ierror, lun,           & ! error flags, logical unit
  tcs_len,                      & ! length of top csize
  bcs_len,                      & ! length of mottom csize 
  i, j, k, ii, jj, kk,          & ! loop variables
  xind, yind, zind,             & ! indices for centroid positions
  size_out,                     & ! size overlaps utility 
  icount, mcs_area_count,       & ! counter
  vals, vals_2d,                & ! buffer size (elements) for allreduce
  iglob, jglob,                 & ! global indizes
  izx, izy,                     & ! utility position indices
  xix, xiy, xiz,                & ! utility position indices
  search_range,                 & ! for area where max is searched
  mcs_search_range               

REAL (KIND=ireals)  ::  &
  grau_max,             &         ! Maximum of 1D slice thru QG centroid
  riming_max,           &         ! Maximum of 1D slice thru GRAR centroid
  graupel_max_scal,     &
  ut_var 

! Logical variables
! -----------------

LOGICAL               ::  &
  print_field

! Variables for debugging
! -----------------------

REAL (KIND=ireals)  ::      &
  deltat1, tmark2, tmark1,  &
  deltat2, tmark3, tmark4,  &
  mg_sum, mg_avg,           &  ! Sum and average of graupel mass per region
  mg_max                       ! Max of graupel mass 

!============================================================================
! Start routine
!============================================================================

!----------------------------------------------------------------------------
! Section 1: Label clusters and analyze their properties
!----------------------------------------------------------------------------

! Label bottom clusters and find centroid positions

bottom_lab = 0.0_ireals
cs_count   = 0_iintegers
bcsize     = 0_iintegers
bottom_lab = 0_iintegers

CALL label (bottom, bottom_lab, bcsize)

bcs_len = cs_count

ALLOCATE(info_bottom       (bcs_len), STAT=istat)
ALLOCATE(bottom_centroid_x (bcs_len), STAT=istat)
ALLOCATE(bottom_centroid_y (bcs_len), STAT=istat)
ALLOCATE(bottom_centroid_z (bcs_len), STAT=istat)

IF (bcs_len > 0) THEN

  CALL cluster_analysis(info_bottom, bottom_lab, bcsize)

  DO ii = 1, bcs_len
    bottom_centroid_x(ii) = info_bottom(ii) % cent_pos_x
    bottom_centroid_y(ii) = info_bottom(ii) % cent_pos_y
    bottom_centroid_z(ii) = info_bottom(ii) % cent_pos_z
  ENDDO

  ! Label top clusters

  top_lab = 0_iintegers
  
  CALL label (top, top_lab, tcsize)
  tcs_len = cs_count

  ALLOCATE(info_top(tcs_len), STAT=istat)
  IF (tcs_len > 0) THEN
    CALL cluster_analysis(info_top, top_lab, tcsize)
  ENDIF

  size_out = MAX(bcs_len, tcs_len)    ! overlap, utility

  ALLOCATE(overlap_top_ut(size_out), STAT = istat)
  ALLOCATE(overlap_bot_ut(size_out), STAT = istat)

  overlap_top_ut = 0_iintegers
  overlap_bot_ut = 0_iintegers

! The labeled field (upper plate) is all-reduced to all processors,
! and hence of dimension (ie_tot, je_tot, ke).  This way, it can
! be accessed with global indices.  Doing it the other way round,
! i.e., finding the local (i,j) and the respective local PE, of
! the bottom centroids would be utterly inefficient.

  rhfl(:,:,1:ke) = 0.5 * (hhl(:,:,2:ke+1) + hhl(:,:,1:ke))

  rho_glob_temp   = 0.0_ireals
  rhfl_glob       = 0.0_ireals
  top_lab_glob    = 0_iintegers
  bottom_lab_glob = 0_iintegers
  qg_glob_temp    = 0.0_ireals

  y_glob = global_j

  DO k = 1, ke
    DO j = jstartpar, jendpar
      DO i = istartpar, iendpar
        iglob = i
        jglob = y_glob(j)
        top_lab_glob   (iglob, jglob, k) = top_lab   (i,j,k)
        bottom_lab_glob(iglob, jglob, k) = bottom_lab(i,j,k)
        rhfl_glob      (iglob, jglob, k) = rhfl      (i,j,k)
        qg_glob_temp   (iglob, jglob, k) = qg        (i,j,k,nnow) 
        rho_glob_temp  (iglob, jglob, k) = rho       (i,j,k)
      ENDDO
    ENDDO
  ENDDO

  rho_glob            = 0.0_ireals
  grar_glob           = 0.0_ireals 
  top_lab_glob_red    = 0_iintegers
  bottom_lab_glob_red = 0_iintegers
  rhfl_glob_red       = 0.0_ireals  
  qg_glob             = 0.0_ireals

  vals    = ie_tot * je_tot * ke_tot
  vals_2d = ie_tot * je_tot 

  CALL MPI_ALLREDUCE                                       &
     (top_lab_glob, top_lab_glob_red, vals,                &
     imp_integers, MPI_SUM, icomm_cart, ierror)

  CALL MPI_ALLREDUCE                                       &
     (bottom_lab_glob, bottom_lab_glob_red, vals,          &
     imp_integers, MPI_SUM, icomm_cart, ierror)

  CALL MPI_ALLREDUCE                                       &
   (rhfl_glob, rhfl_glob_red, vals,                        &
    imp_reals, MPI_MAX, icomm_cart, ierror)

  CALL MPI_ALLREDUCE                                       &
   (grar_glob_temp, grar_glob, vals,                       &
   imp_reals, MPI_MAX, icomm_cart, ierror)

  CALL MPI_ALLREDUCE                                       &
   (rho_glob_temp, rho_glob, vals,                         &
   imp_reals, MPI_MAX, icomm_cart, ierror)

  CALL MPI_ALLREDUCE                                       &
   (qg_glob_temp, qg_glob, vals,                           &
   imp_reals, MPI_MAX, icomm_cart, ierror)

  ! Find label of upper pair member

  icount = 0_iintegers
  cs_loop: DO ii = 1, bcs_len
    xind = bottom_centroid_x(ii)
    yind = bottom_centroid_y(ii)
    zind = bottom_centroid_z(ii)
    k_loop: DO k = zind, 1, -1  ! going upward
      IF (top_lab_glob_red(xind, yind, k) /= 0) THEN 
        icount = icount + 1_iintegers
        overlap_top_ut(icount) = top_lab_glob_red(xind, yind, k)
        overlap_bot_ut(icount) = info_bottom(ii) % label
        EXIT k_loop
      ENDIF
    ENDDO k_loop
  ENDDO cs_loop

  number_overlaps = icount 
 
  ! Check if size of info_cap structure is sufficient
  
  IF (number_overlaps > max_number_cap) THEN
    IF (my_cart_id == 0) THEN
      print *, ' *** SRC_LIGHTNING.CLUSTER_OVERLAPS ERROR ***'
      print *, 'CLUSTER_OVERLAPS: Capacitor structure assigned too few elements.'
      print *, 'Increase MAX_NUMBER_CAP to at least', number_overlaps
    ENDIF  
    CALL MPI_ABORT(MPI_COMM_WORLD, 100, ierror)
  ENDIF

  ! Obtain structure contents and store them in arrays for better access 
  ! (vectorization)

  ! Initialize and assign capacitor structure

  info_cap % cap_label      = 0_iintegers
  info_cap % top_label      = 0_iintegers
  info_cap % top_pixels     = 0_iintegers
  info_cap % top_area       = 0.0_ireals
  info_cap % top_depth      = 0.0_ireals
  info_cap % separation     = 0.0_ireals
  info_cap % plate_distance = 0.0_ireals
  info_cap % breakdown_alt  = 0.0_ireals
  info_cap % x_pos          = 0_iintegers
  info_cap % y_pos          = 0_iintegers
  info_cap % bot_label      = 0_iintegers
  info_cap % bot_pixels     = 0_iintegers
  info_cap % bot_area       = 0.0_ireals
  info_cap % bot_depth      = 0.0_ireals
  info_cap % graupel        = 0.0_ireals
 
   DO ii = 1, number_overlaps
    IF (overlap_top_ut(ii) /= 0) THEN
      info_cap(ii) % cap_label  = ii
      info_cap(ii) % top_label  = overlap_top_ut(ii) 
      info_cap(ii) % top_pixels = info_top(overlap_top_ut(ii)) % pixels  
      info_cap(ii) % x_pos      = info_bottom(overlap_bot_ut(ii)) % cent_pos_x
      info_cap(ii) % y_pos      = info_bottom(overlap_bot_ut(ii)) % cent_pos_y 
      info_cap(ii) % bot_label  = overlap_bot_ut(ii)
      info_cap(ii) % bot_pixels = info_bottom(overlap_bot_ut(ii)) % pixels
    ENDIF 
  ENDDO
 
  ! Calculate "strength" of lower plate (GRAR, QG,...)

  DO ii = 1, number_overlaps

    izx = info_cap(ii) % x_pos
    izy = info_cap(ii) % y_pos 
     
    search_range    = 2_iintegers
    graupel_max_arr = 0.0_ireals

    IF (izx > search_range .AND. xind < ie - search_range .AND. &
        izy > search_range .AND. yind < je_tot - search_range) THEN
      DO k = 1, ke
        DO j = izy-search_range, izy+search_range
          DO i = izx-search_range, izx+search_range  
            graupel_max_arr(i,j,k) = rho(i,j,k) * qg_glob(i,j,k)  ! Adjust here for density
          ENDDO
        ENDDO
      ENDDO
      grau_max = MAXVAL(graupel_max_arr(:,:,:))
    ENDIF
 
    info_cap(ii) % graupel = graupel_mass_correction * grau_max

  ENDDO

  ! Determine the breakdown altitude, i.e., the altitude between the plates.
  ! This is just the average of the upper and lower z-centroid positions.
  ! Though the centroids may be strongly offset laterally, this estimate
  ! should be OK.

  DO ii = 1, number_overlaps
    xiz = 0.5_ireals * (info_top(ii)%cent_pos_z + info_bottom(ii)%cent_pos_z)
    xiy = info_cap(ii) % y_pos
    xix = info_cap(ii) % x_pos
    info_cap(ii) % breakdown_alt = rhfl_glob_red(xix, xiy, xiz)   
  ENDDO

  ! Complete entries of info_cap structure
 
  CALL capacitor_details  (info_top, info_bottom, &
                           top_lab_glob_red, bottom_lab_glob_red,   &
                           number_overlaps, rhfl_glob_red)

ELSEIF (bcs_len == 0) THEN

  info_cap % cap_label     = 0_iintegers
  info_cap % top_label     = 0_iintegers
  info_cap % top_pixels    = 0_iintegers
  info_cap % top_area      = 0.0_ireals
  info_cap % top_depth     = 0.0_ireals
  info_cap % separation    = 0.0_ireals
  info_cap % x_pos         = 0_iintegers
  info_cap % y_pos         = 0_iintegers
  info_cap % bot_label     = 0_iintegers
  info_cap % bot_pixels    = 0_iintegers
  info_cap % bot_area      = 0.0_ireals
  info_cap % bot_depth     = 0.0_ireals
  info_cap % graupel       = 0.0_ireals
  info_cap % breakdown_alt = 0.0_ireals

ENDIF

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(bottom_centroid_x, STAT = istat)
DEALLOCATE(bottom_centroid_y, STAT = istat)
DEALLOCATE(bottom_centroid_z, STAT = istat)
DEALLOCATE(info_top,          STAT = istat)
DEALLOCATE(info_bottom,       STAT = istat)
DEALLOCATE(overlap_top_ut,    STAT = istat)
DEALLOCATE(overlap_bot_ut,    STAT = istat)

!============================================================================
! End subroutine cluster_overlaps
!============================================================================

END SUBROUTINE cluster_overlaps

!============================================================================
! Subroutine capacitor_details
!============================================================================

!----------------------------------------------------------------------------
! Description:
!   This internal module procedure calculates the area as well as the depth
!   of the overlapping clusters.  Also, the separatio distancen is calculated. 
!
! Method:
!  The horizontal area: Calculated based on level of the cluster centroid.
!  A horizontal slice through the centroid is considered and the pixels in
!  this slice are counted.  Since the area size is known, the area of the
!  entire cluster follows trivially.  The rotated grid results in squares
!  quite closely, but still, dlon is scaled with R*COS(phi).  phi is taken
!  at the centroid of the respective cell whose area is calculated.
!  
!  The vertical depth: Calculated by considering a vertical 1D slab 
!  through the centroid.  The number of occupied levels is counted, and
!  since the vertical grid spacing is known, the depth of the cluster 
!  follows.  This is achieved with the help of the HHL field. 
!
! Note:
!  This is serial code, run on processor number zero.
!----------------------------------------------------------------------------

SUBROUTINE capacitor_details (zinfo_top, zinfo_bottom,   &
                              global_top, global_bottom, &
                              znumber_overlaps, zhfl)

!----------------------------------------------------------------------------
! Subroutine arguments
!----------------------------------------------------------------------------

IMPLICIT NONE

! 1D structures
! -------------

TYPE(cluster_info), INTENT(IN)  ::  &
  zinfo_top   (:),      &      ! cluster statistics of top cluster
  zinfo_bottom(:)              ! cluster statistics of bottom cluster 

!TYPE(capacitor), INTENT(INOUT)  ::  &
!  zinfo_capa(:)               ! capacitor-info structure

! 3D fields
! ---------

INTEGER (KIND=iintegers), INTENT(IN)  ::  &
  global_top   (ie_tot, je_tot, ke_tot),  &  ! global upper field
  global_bottom(ie_tot, je_tot, ke_tot)      ! global bottom field

REAL (KIND=ireals),  INTENT(IN)  ::       &
  zhfl(ie_tot, je_tot, ke_tot)               ! height of full levels

! 2D fields
! ---------

! 1D fields
! ---------

INTEGER (KIND=iintegers),  INTENT(IN) ::  &
  znumber_overlaps                          ! number of overlapping clusters

!----------------------------------------------------------------------------
! Local variables
!----------------------------------------------------------------------------

! Local parameters
! ----------------

! None

! Local scalars
! -------------

REAL (KIND=ireals)  :: &
  zlat, zlon,          &          ! lat/lon at cell centroid 
  dx, dy,              &          ! grid spacing
  futility                        ! utility floating point number 

INTEGER (KIND=iintegers)  ::  &
  i, j, k, ii, jj, kk,        &  ! loop variables
  zerror, zstat,              &  ! error/status flags                     
  zbot_label,                 &  ! label of lower cluster
  zcount,                     &  ! counter variable
  lzrs,                       &  ! scalar version of zrs
  xpos, ypos,                 &  ! position of cell centroid
  iutility1, iutility2,       &  ! utility integers
  ztop_label,                 &  ! label of top cluster
  vals_2d,                    &  ! buffer size for global reduction
  iglob, jglob,               &  !
  zvdist,                     &  ! distance of plates
  bot_cent                       ! INT of bottom centroid height

! Local 1D arrays
! ---------------

INTEGER (KIND=iintegers) ::  &
  x_glob(ie),           &
  y_glob(je)

! Local dynamic 1D arrays
! -----------------------

INTEGER (KIND=iintegers), ALLOCATABLE  ::  &
  area_bot_pixels(:),                      & ! pixels in horizontal slice
  summit_bottom(:),                        & ! top height of bottom cluster 
  base_bottom(:),                          & ! base height of bottom cluster
  summit_top(:),                           & ! top height of upper cluster 
  base_top(:)                                ! base height of upper cluster

! Local 2D arrays
! ---------------

INTEGER (KIND=iintegers)  ::          &
  utility_field_bot(ie_tot, je_tot) 
  
!REAL (KIND=ireals) ::                  &
!  qg_slice_glob_temp(ie_tot, je_tot),  &
!  qg_slice_glob     (ie_tot, je_tot),  & 
!  qg_slice_local    (ie,     je    )

! Local 3D arrays
! ---------------

! None

! Debug variables
! ---------------

INTEGER (KIND=iintegers)  ::  &
  lun

!----------------------------------------------------------------------------
! Allocate dynamic arrays
!----------------------------------------------------------------------------

ALLOCATE(area_bot_pixels(znumber_overlaps), STAT=zstat)
ALLOCATE(summit_bottom  (znumber_overlaps), STAT=zstat)
ALLOCATE(base_bottom    (znumber_overlaps), STAT=zstat)
ALLOCATE(summit_top     (znumber_overlaps), STAT=zstat)
ALLOCATE(base_top       (znumber_overlaps), STAT=zstat)

!============================================================================
! Start routine
!============================================================================

! Initialize arrays

summit_bottom = 0.0_ireals
base_bottom   = 0.0_ireals
summit_top    = 0.0_ireals
base_top      = 0.0_ireals

!----------------------------------------------------------------------------
! Section 1: Find area and depth of the "lower plate"
!----------------------------------------------------------------------------

! Count pixels of slab through centroid

zrs = 0_iintegers

DO ii = 1, znumber_overlaps

  ! Calculate area: Count pixels
  ! ----------------------------

  ztop_label = info_cap(ii) % top_label
  zbot_label = info_cap(ii) % bot_label
  zrs(ii)    = zinfo_bottom(zbot_label) % cent_pos_z  ! Module variable 
  lzrs       = zrs(ii)
  zcount     = 0_iintegers
  utility_field_bot(:,:) = global_bottom(:,:,lzrs)

  DO j = 1, je_tot
    DO i = 1, ie_tot
      IF (utility_field_bot(i,j) == zbot_label) THEN
        zcount = zcount + 1
      ENDIF
    ENDDO
  ENDDO

  area_bot_pixels(ii) = zcount
  
  ! calculate the area
  ! ------------------

  xpos = zinfo_bottom(zbot_label) % cent_pos_x
  ypos = zinfo_bottom(zbot_label) % cent_pos_y

  zlat = startlat_tot + dlat * (ypos - 1_iintegers)

  dx = r_earth * COS(zlat * pi/180.0_ireals) * dlon * pi/180.0_ireals
  dy = r_earth * dlat * pi/180.0_ireals    
   
  ! Witdh scaling factor  
 
  info_cap(ii) % bot_area = &
  storm_width_correction * area_bot_pixels(ii) * dx * dy

  ! Calculate depth (lower plate)
  ! -----------------------------

  downward_loop: DO k = 1, ke
    IF (global_bottom(xpos, ypos, k) == zbot_label) THEN
      summit_bottom(ii) = zhfl(xpos, ypos, k) 
      EXIT downward_loop
    ENDIF
  ENDDO downward_loop

  upward_loop: DO k = ke, 1, -1   
    IF (global_bottom(xpos, ypos, k) == zbot_label) THEN
      base_bottom(ii) = zhfl(xpos, ypos, k) 
      EXIT upward_loop
    ENDIF 
  ENDDO upward_loop

  info_cap(ii) % bot_depth = summit_bottom(ii) - base_bottom(ii)
   
  !--------------------------------------------------------------------------
  ! Section 2: Calculate area and depth of upper plate 
  !
  !  A projection of the lower plate onto the upper plate is used.  In
  !  this case, the projection is one-to-one, i.e., the area is equal
  !  to that of the lower plate.  The upper-plate depth is calculated based
  !  on the centroid position of the lower plate.
  !--------------------------------------------------------------------------

  info_cap(ii) % top_area   = info_cap(ii) % bot_area 
  info_cap(ii) % top_pixels = area_bot_pixels(ii)

  ! Calculate depth (upper plate)
  ! -----------------------------

  loop_down: DO k = 1, ke
    IF (global_top(xpos, ypos, k) == ztop_label) THEN
      summit_top(ii) = zhfl(xpos, ypos, k)
      EXIT loop_down
    ENDIF
  ENDDO loop_down
 
  info_cap(ii) % total_height = summit_top(ii)
 
  loop_up: DO k = ke, 1, -1
    IF (global_top(xpos, ypos, k) == ztop_label) THEN
      base_top(ii) = zhfl(xpos, ypos, k)  
      EXIT loop_up 
    ENDIF
  ENDDO loop_up

  info_cap(ii) % top_depth = summit_top(ii) - base_top(ii) !! Added later
 
  ! Calculate vertical separation distance
  ! --------------------------------------

  zvdist = base_top(ii) - summit_bottom(ii)

  ! Vertical separation distance is equal to the centroid distance.

  info_cap(ii) % separation = REAL(zvdist)

  IF (zvdist <= 0) THEN   ! Overlap

    ! Round off 2nd term but keep it real

    bot_cent = INT(zhfl(xpos, ypos, zinfo_bottom(ii) % cent_pos_z))

    info_cap(ii) % plate_distance = &
    REAL(summit_bottom(ii) - bot_cent)

    info_cap(ii) % separation = 0.0_ireals
    
  ELSEIF (zvdist > 0) THEN

    info_cap(ii) % plate_distance =                &
    zhfl(xpos, ypos, zinfo_top(ii) % cent_pos_z) - &
    zhfl(xpos, ypos, zinfo_bottom(ii) % cent_pos_z)

  ENDIF

  ! IF the distance still turns out to be (smaller than) zero, 
  ! set it to 500 m     

  IF (info_cap(ii) % plate_distance <= 0.0) THEN
    info_cap(ii) % plate_distance = 500.0_ireals
    IF (my_cart_id == 0) THEN
      WRITE (*,*) 'ATTENTION: Separation distance error: ', ntstep, ii, &
      info_cap(ii) % plate_distance 
    ENDIF
  ENDIF

  ! One-pixel cells have depth zero and cause an exception farther above.
  ! The following avoids that.

  IF (info_cap(ii) % top_depth < 1.E-5) THEN
    info_cap(ii) % top_depth = 1.E-2
  ENDIF
  IF (info_cap(ii) % bot_depth < 1.E-5) THEN
    info_cap(ii) % bot_depth = 1.E-2
  ENDIF

ENDDO ! ii-loop through clusters

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(area_bot_pixels, STAT=zstat)
DEALLOCATE(summit_bottom  , STAT=zstat)
DEALLOCATE(base_bottom    , STAT=zstat)
DEALLOCATE(summit_top     , STAT=zstat)
DEALLOCATE(base_top       , STAT=zstat)

!============================================================================
! End subroutine capacitor_details
!============================================================================

END SUBROUTINE capacitor_details

!============================================================================
! Subroutine distribute_flashes
!============================================================================

!----------------------------------------------------------------------------
! Description:
!   Based on the location of the cell centroid (i,j) as well as their
!   instantaneous lightning frequencies, three arrays are created, containing
!   flash locations (lon, lat in rotated coordinates) as well as the times 
!   of their occurrence.  
!   The spatial distribution is circularly-symmetric about the cell centroid
!   position.  Upstream distribution has been attempted but yielded no
!   satisfactory results.  The optimum seems to be a circular distribution
!   where the radius corresponds to the width of the cell.
!
! Method:
!   A random-number generator is used to distribute the flashes. For the 
!   spatial distribution, a Gauss-weighing is applied, while the temporal
!   distribution is purely random.
!  
!   The time step as well as the interval when the lightning package are 
!   called, determine the time interval across which the flashes have to
!   be distributed.  The instantaneous flash rate is given in s^(-1), so
!   the number of flashes to be distributed is simply the calling interval
!   (in steps) times the step size times the flash frequency.
!
!    Those cells that exhibit too weak a lightning rate to result in
!    at least one lightning per period are included but their lon/lat
!    couplet is set to -999.  These values must be filtered by the
!    post-processing software. 
!----------------------------------------------------------------------------

SUBROUTINE distribute_flashes (dcell, n_tot, lon, lat, time, bot_labeled)

!----------------------------------------------------------------------------
! Subroutine arguments
!----------------------------------------------------------------------------

IMPLICIT NONE

! INTENT(IN)

TYPE(storm), INTENT(IN)    ::  &
  dcell(:) ! (cs_count) Assumed size; either cs_count or number_overlaps

INTEGER (KIND=iintegers)        ::  &
  n_tot                                 ! total number of strikes per
                                        ! hinclight
INTEGER (KIND=iintegers) :: &
  bot_labeled(ie,je,ke)

! INTENT(OUT)

REAL (KIND=ireals), INTENT(OUT)      ::  &   ! Assumed shape from call
  lon (:),                               &
  lat (:),                               &
  time(:)

!----------------------------------------------------------------------------
! Local variables
!----------------------------------------------------------------------------

! Tempo variables of various kinds for debugging purposes
! -------------------------------------------------------

REAL (KIND=ireals)  :: &
  tmark1, tmark2, tmark3, tmark4, deltat1, deltat2

! Local parameters
! ----------------

REAL (KIND=ireals), PARAMETER :: &
  min_width  = 0.17_ireals        ! 3 GPs width of lightning path: 0.075 deg
                                  !  width_min  = 0.17 looks good for some reason   

! Local scalars
! -------------

! DECL  (mark)

REAL (KIND=ireals)        ::  &
  sinclight,                  &       ! hinclight in seconds
  flash_freq,                 &
  squeeze,                    &
  deg2rad, rad2deg,           &
  cent_x_angl,                & ! x-position of centroid in COSMO coordinates
  cent_y_angl,                & ! y-position of centroid in COSMO coordinates
  cent_x_angl_geo,            & ! Same but in geographical coordinates
  cent_y_angl_geo,            & ! Same but in geographical coordinates
  llsigma, weighfac,          & ! coefficients of weighing function
  bunkers_mean,               & ! mean wind in lowest ~6km
  path_length_km,             & ! length of cell's path during hinclight
  path_length_ang,            & ! distance expressed in degrees 
  ltheta,                     & ! azimuth angle
  sdi2_centroid,              & ! SDI at cell centroid position
  c_magn,                     & ! storm motion vector length 
  width,                      & ! diameter of path/circles 
  cy, cx,                     & ! Storm motion vector
  um_vert, vm_vert,           & ! (faked) vertically-averaged area mean
  shear_magn,                 &
  u_bulk_shear,               &
  v_bulk_shear,               &
  dangle,                     & 
  equiv_diam_km                 ! Equivalent circular cell diameter

INTEGER (KIND=iintegers)  ::  &
  i, ii, j, jj, k, kk,        &      ! loop variables
  dstat, derror, izerror,     &
  lun, lc_count,              &
  iglob, jglob,               &
  xind, yind, x,              &
  steering_height_ind,        &      ! k of breakdown alt
  area_count, vert_count             ! 2 counters

INTEGER (KIND=iintegers)  ::  &
  t_current, t_start,         & ! current time and time at beginning of interval
  n_cell,                     & ! flashes per cell and interval
  arr_start, arr_end,         & ! start/end indices for output array concatenation 
  ngp,                        & ! number of gridopints (width of lightning trace) 
  vals,                       & ! number of elements of global 2D field
  vals3d,                     & !
  avg_range,                  & ! gridpoint number for horizontal mean
  flash_margin

! Local arrays
! ------------

! 1D arrays
! ---------

INTEGER (KIND=iintegers)   ::  &
  n_cell_arr(cs_count+1),      &  ! array containing the number of flashes per 
                                  ! cell per time interval (contains 1 as 
                                  ! first element)
  x_arr(450),                  &          
  iglobind(ie), jglobind(je),  & 
  test_var(30)

REAL (KIND=ireals) ::       &
  um_area(ke), vm_area(ke)       ! area means

! 2D arrays
! ---------
  
REAL (KIND=ireals)         ::   &
  sdi_1          (ie,je),        &
  sdi_2          (ie,je),         &
  sdi_2_glob_temp(ie_tot, je_tot), &
  sdi_2_glob     (ie_tot, je_tot)

! 3D arrays
! ---------

REAL (KIND=ireals) ::                      &
  u_glob_temp    (ie_tot, je_tot, ke_tot), &
  v_glob_temp    (ie_tot, je_tot, ke_tot), &
  u_glob         (ie_tot, je_tot, ke_tot), &
  v_glob         (ie_tot, je_tot, ke_tot), &
  hfl_glob_temp  (ie_tot, je_tot, ke_tot)

REAL (KIND=ireals)  ::  &
  hfl(ie,je,ke)

! Local dynamic arrays
! --------------------

INTEGER (KIND=iintegers), ALLOCATABLE  :: &
  argument(:),                   &     ! independent variables for Gauss
  li_cell_arr(:),                &     ! same as n_cell_arr but w/o zeros
  xpos_cell_arr(:),              &     ! x-coordinates of flashing cells
  ypos_cell_arr(:)                     ! y-coordinates of flashing cells
  
REAL (KIND=ireals), ALLOCATABLE  :: &
  weighing(:),                      &  ! Gauss function
  dist(:),                          &  ! radial distance 
  angl(:),                          &  ! azimuth angle
  weighted(:),                      &  ! weighted distance  
  x_loc(:),                         &  ! x-coord of flash rel to centroid
  y_loc(:),                         &  ! y-coord of flash
  x_loc_final(:),                   &  ! x-coord in COSMO coordinates
  y_loc_final(:),                   &  ! y-coord in COSMO coordinates
  xx(:),                            &  ! tempo x-coord in "rectangle system"
  yy(:),                            &
  yy_ut(:),                         &  ! Utility field for random numbers
  x_rot(:), y_rot(:),               &
  x_shifted(:), y_shifted(:),       &
  pwidth(:),                        &
  pwidth_flashing(:),               & 
  steering_alt(:)

! Characters
! ----------

CHARACTER (LEN=3) ::  &
  time_ut_arr1

CHARACTER (LEN=4) ::  &
  time_ut_arr2

CHARACTER (LEN=5) ::  &
  time_arr,           &
  time_ut_arr3

LOGICAL           ::  &
  printdb = .FALSE.   
                    
!============================================================================
! Start procedure
!============================================================================

IF (itype_light == 1 .AND. cs_count < number_overlaps .AND. my_cart_id == 0) THEN
  print *, 'SRC_LIGHTNING.DISTRIBUTE_FLASHES ERROR'
  print *,'Inconsistent number of cells:', cs_count, number_overlaps
  print *, 'Accum. flashes, time step', n_tot, ntstep
ENDIF

IF (itype_light == 1) THEN
  cs_count = number_overlaps      ! Adjust trip counts for 2-class cell
ENDIF                             ! global var from LABEL overwritten

deg2rad = pi/180.0_ireals
rad2deg = 180.0_ireals/pi

sinclight = lightning_step * dt     ! calling interval in seconds
t_current = ntstep * dt             ! seconds after initialization
t_start   = t_current - sinclight

!----------------------------------------------------------------------------
! Section 1.1: Determine the width of cells
!----------------------------------------------------------------------------

ALLOCATE (pwidth(cs_count), STAT = dstat); pwidth = 0.0_ireals

!--------------------------------------------------------------------------
! Section 1.2: Prepare fields needed for determining storm motion 
!--------------------------------------------------------------------------

hfl(:,:,1:ke) = 0.5_ireals * (hhl(:,:,1:ke) + hhl(:,:,2:ke+1))

!----------------------------------------------------------------------------
! Section 2: Continue serially on processor #0 - distribute flashes
!----------------------------------------------------------------------------

IF (my_Cart_id == 0) THEN

  IF (n_tot == 0) THEN
    time = -999.0_ireals       ! Arrays have two elements (as allocated above)
    lon  = -999.0_ireals
    lat  = -999.0_ireals
  ELSEIF (n_tot > 0) THEN

    !--------------------------------------------------------------------------
    ! Section 2.1:
    !  Create times of occurrence in seconds after initialization
    !--------------------------------------------------------------------------
  
    CALL RANDOM_SEED
    CALL RANDOM_NUMBER (time)

    time = t_start + (sinclight * time)

    !--------------------------------------------------------------------------
    ! Section 2.2:
    ! Distribute flashes spatially
    !--------------------------------------------------------------------------

    ! Array containing the number of flashes per call interval and cell are
    ! calculated in a separate loop 

    n_cell_arr    = 0_iintegers
    DO ii = 1, cs_count
      flash_freq = dcell(ii) % flash_rate
      n_cell =   NINT( flash_freq * 3600.0_ireals * hinclight )
      n_cell_arr(ii+1) = n_cell
    ENDDO

    ! Initialize with dummy values

    lon = -999.0_ireals
    lat = -999.0_ireals

    ! Create new n_cell array and cs_count, containing only those cells
    ! that actually produce lightning.  First loop is to find length
    ! of lightning-cell array, second loop is for assignment.

    lc_count = 0_iintegers
    DO ii = 2, cs_count+1
      IF (n_cell_arr(ii) /= 0) THEN
        lc_count = lc_count + 1
       ENDIF 
    ENDDO

    ALLOCATE (li_cell_arr(lc_count+1)  , STAT = dstat)
    ALLOCATE (xpos_cell_arr(lc_count)  , STAT = dstat)
    ALLOCATE (ypos_cell_arr(lc_count)  , STAT = dstat)
    ALLOCATE (pwidth_flashing(lc_count), STAT = dstat)
    ALLOCATE(steering_alt(lc_count)    , STAT = dstat)

    li_cell_arr   = 0_iintegers
    xpos_cell_arr = 0_iintegers
    ypos_cell_arr = 0_iintegers

    lc_count    = 0_iintegers

    DO ii = 2, cs_count+1
      IF (n_cell_arr(ii) /= 0) THEN
        lc_count = lc_count + 1
        li_cell_arr(lc_count+1) = n_cell_arr(ii)
        xpos_cell_arr(lc_count) = dcell(ii-1) % centroid_x        
        ypos_cell_arr(lc_count) = dcell(ii-1) % centroid_y
        steering_alt(lc_count)    = zrs(ii-1)
      ENDIF
    ENDDO

    arr_start = 1_iintegers
    arr_end   = 0_iintegers

    ! Loop over flashing cells
    ! ------------------------

    DO ii = 1, lc_count

      n_cell = li_cell_arr(ii+1) 

      ALLOCATE(argument   (n_cell),  STAT = dstat)
      ALLOCATE(weighing   (n_cell),  STAT = dstat)
      ALLOCATE(dist       (n_cell),  STAT = dstat) 
      ALLOCATE(angl       (n_cell),  STAT = dstat)
      ALLOCATE(weighted   (n_cell),  STAT = dstat)
      ALLOCATE(x_loc      (n_cell),  STAT = dstat)
      ALLOCATE(y_loc      (n_cell),  STAT = dstat)
      ALLOCATE(x_loc_final(n_cell),  STAT = dstat)
      ALLOCATE(y_loc_final(n_cell),  STAT = dstat)
      ALLOCATE(xx         (n_cell),  STAT = dstat)
      ALLOCATE(yy         (n_cell),  STAT = dstat)
      ALLOCATE(yy_ut(3_iintegers * n_cell),  STAT = dstat)
      ALLOCATE  (x_shifted(n_cell),  STAT = dstat)
      ALLOCATE  (y_shifted(n_cell),  STAT = dstat)
      ALLOCATE      (x_rot(n_cell),  STAT = dstat)
      ALLOCATE      (y_rot(n_cell),  STAT = dstat)

      !----------------------------------------------------------------------
      ! Section 2.4:
      ! Distribute flashes in a circle around the cell      
      !----------------------------------------------------------------------
     
      ! Diameter of the cell
       
      equiv_diam_km = 2.0_ireals * SQRT(1.E-6 * info_cap(ii) % bot_area / pi) 
 
      ! Form-preserving weighing (within [0, n_cell] interval)
      ! 0 <= weighing(x) < 1 for x eps [0, ncell]

      llsigma    = 0.4_ireals * n_cell
      weighfac = - 1.0_ireals / (2.0_ireals * (llsigma * llsigma))

      ! Create data in polar coordinates and thereafter transform into
      ! Cartesian grid

      IF (itype_light == 1) THEN
        pwidth_flashing(ii) = 0.5_ireals * equiv_diam_km / &       ! Radius 
                              (r_earth * 1.0E-3_ireals) * rad2deg 
      ELSEIF (itype_light > 1) THEN
        pwidth_flashing(ii) = 0.05_ireals + 2.5E-4_ireals * n_cell 
      ENDIF

      IF (pwidth_flashing(ii) <= min_width) pwidth_flashing(ii) = min_width

      IF (printdb) WRITE (*,*) 'TRACK_WIDTH', ii, pwidth_flashing(ii)

      CALL RANDOM_SEED
      CALL RANDOM_NUMBER (dist)

      dist = pwidth_flashing(ii) * dist 
      
      ! Create uncorrelated random-number series
   
      CALL RANDOM_SEED
      CALL RANDOM_NUMBER (yy_ut)
  
      angl(1:n_cell) = yy_ut(3:n_cell+2)
   
      angl = 2.0_ireals * pi * angl      ! in radians

      ! Set up Gauss function; starting from x = 0
   
      argument = 0_iintegers
      DO i = 2, n_cell
        argument(i) =  argument(i) + (i-1)
      ENDDO
      argument(1) =  0_iintegers
   
      weighing = EXP(weighfac * (REAL(argument) * REAL(argument)))
  
      weighted = dist * weighing
 
      ! Cartesian coordinates

      x_loc = weighted * COS(angl)
      y_loc = weighted * SIN(angl)

      cent_x_angl = startlon_tot + dlon *  (xpos_cell_arr(ii) - 1) 
      cent_y_angl = startlat_tot + dlat *  (ypos_cell_arr(ii) - 1)

      x_loc_final = x_loc + cent_x_angl
      y_loc_final = y_loc + cent_y_angl

      arr_start = arr_start + li_cell_arr(ii)
      arr_end   = arr_end   + li_cell_arr(ii+1)

      ! Concatenate into final array

      lon(arr_start:arr_end) = x_loc_final
      lat(arr_start:arr_end) = y_loc_final 

      DEALLOCATE(argument   , STAT = dstat)
      DEALLOCATE(weighing   , STAT = dstat)
      DEALLOCATE(dist       , STAT = dstat)
      DEALLOCATE(angl       , STAT = dstat)
      DEALLOCATE(weighted   , STAT = dstat)
      DEALLOCATE(x_loc      , STAT = dstat)
      DEALLOCATE(y_loc      , STAT = dstat)
      DEALLOCATE(x_loc_final, STAT = dstat)
      DEALLOCATE(y_loc_final, STAT = dstat)
      DEALLOCATE(xx         , STAT = dstat)
      DEALLOCATE(yy         , STAT = dstat)
      DEALLOCATE(yy_ut      , STAT = dstat)
      DEALLOCATE(x_shifted  , STAT = dstat)
      DEALLOCATE(y_shifted  , STAT = dstat)
      DEALLOCATE(x_rot      , STAT = dstat)
      DEALLOCATE(y_rot      , STAT = dstat)

    ENDDO    ! loop over flashing cells
  ENDIF    ! ntot > 0
ENDIF    ! my_cart_id == 0  

!----------------------------------------------------------------------------
! Section 3:
!   Write output as ASCII
!   files:
!
!   The target directory is: 
!     /e/uhome/dlrdahl/lightning_output/
!
!   File-name format:
!     lightning_sinclight.txt
! 
!----------------------------------------------------------------------------

IF (my_cart_id == 0) THEN

  IF (t_current >= 1E2 .AND. t_current < 1E3) THEN
    WRITE(time_arr,'(i3)') INT(t_current)
    time_arr = '00'// time_arr
  ELSEIF (t_current >= 1E3 .AND. t_current < 1E4) THEN
    WRITE(time_arr,'(i4)') INT(t_current)
    time_arr = '0'//time_arr
  ELSEIF(t_current >= 1E4 .AND. t_current < 1E5) THEN
    WRITE(time_arr,'(i5)') INT(t_current)
    !time_arr = time_ut_arr3
  ENDIF

  CALL get_free_unit(lun)
 
  OPEN &
    (unit = lun, file = &
    '/e/uhome/dlrdahl/lightning_output/lightning_'// time_arr //'.txt')
    WRITE (lun, *)  '       COSMO Lightning Data' 
    WRITE (lun, *)  'TIME           LON          LAT' 
    WRITE (lun, *)  '-----------------------------------'
    WRITE (lun, *)
    
    IF (n_tot > 0) THEN 
      DO i = 1, n_tot ! cs_count
        WRITE(lun, '(3f12.5)') time(i), lon(i), lat(i)
      ENDDO
    ELSE IF (n_tot == 0) THEN
      DO i = 1, 2
        WRITE(lun, '(3f12.5)') time(i), lon(i), lat(i)
      ENDDO
    ENDIF 
  CLOSE (lun)

  CALL release_unit(lun)

ENDIF

!----------------------------------------------------------------------------
! Deallocate dynamic arrays
!----------------------------------------------------------------------------

DEALLOCATE(steering_alt,     STAT = dstat)
DEALLOCATE (pwidth,          STAT = dstat)
DEALLOCATE (li_cell_arr,     STAT = dstat)
DEALLOCATE (xpos_cell_arr,   STAT = dstat) 
DEALLOCATE (ypos_cell_arr,   STAT = dstat)
DEALLOCATE (pwidth_flashing, STAT = dstat)

!============================================================================
! End subroutine distribute_flashes
!============================================================================

END SUBROUTINE distribute_flashes

!============================================================================
! Subroutine LIGHTNING_UTILITIES
!============================================================================

!----------------------------------------------------------------------------
! Purpose:
!  The calling of some functions etc. that are called frequently are moved 
!  to this routine, where module-wide accessible PRIVATE variables are 
!  updated.  The routine needs to be called just once this way.
!
! Method:
!  Simple library calls.
!----------------------------------------------------------------------------

SUBROUTINE lightning_utilities

!----------------------------------------------------------------------------
! Local variables
!----------------------------------------------------------------------------

IMPLICIT NONE 

! local scalars
! -------------

INTEGER (KIND=iintegers)  ::  &
  i, j, k,ii, jj, kk,         &
  ierr, istat

!============================================================================
! Start routine
!============================================================================

! Global indices

DO j = 1, je
  global_j(j) = j_global(j)
ENDDO

! Not currently needed, as the domain composition is such that i == i_glob

! DO i = 1, iend
!   global_i(i) = i_global(i)
! ENDDO

!============================================================================
! End subroutine LIGHTNING_UTILITIES
!============================================================================

END SUBROUTINE lightning_utilities

!============================================================================
! END MODULE LIGHTNING
!============================================================================

END MODULE src_lightning

