;---------------------------------------------------------------------------
; IDL-procedure convert_cosmo.pro
;---------------------------------------------------------------------------

PRO CONVERT_COSMO

;============================================================================
; Description and Method:
;   This procedure post-processes COSMO-DE lightning output.  Three arrays,
;   containing the times of the flashes in seconds after midnight (UTC), 
;   as well as their locations in rotated (i.e., COSMO) coordinates, are
;   used as input.
;   This procedure transforms these data into hh:mm:ss format, as well as
;   lon, lat - and writes them to a file of the same format as the LINET
;   data.
;   Depending on the chosen time interval, several output files from the
;   COSMO will be used and concatenated into one array containing data
;   of the entire interval (e.g., one day).
;
; NOTE:
;   Use of the EXECUTE function is made, and hence this program will run
;   on unix systems, only.
;
;   Some minor inaccuracy exists in the routine that converts seconds to 
;   hh:mm:ss.  This is due to round-off errors, but these only affect the 
;   minutes and seconds - since the times are randomly-generated, this error 
;   is negligible for the current purpose.  
; 
; Calling sequence: Enter convert_cosmo in IDL command line.  
;                   Finish user adjustments before use.
;
; Author: DLR, Johannes Dahl
; phone : +49 (0)8153 28 3556
; email : Johannes.Dahl@dlr.de
;
; History:
; Version    Date       Name
; ---------- ---------- -------------
; 1.1        2010/01/26 Johannes Dahl
;  Initial release
;
;============================================================================

;----------------------------------------------------------------------------
; Prliminaries
;----------------------------------------------------------------------------

Compile_Opt defint32                  ; Set integers to type = long,
                                      ; i.e., 4 bytes (32 bits)
!EXCEPT=2                             ; Print where underflow error occurred.

;============================================================================
; Section I: User adjustments
;============================================================================

; count # of files w/  shell: ls -1 | wc -l 

linet = 1        ; 1: Only consider flashes that occurred in LINET domain

type      = 'D10'       ; type of parameterization; string variable!
hinclight =  0.25       ; calling interval in hours
hend      =  24.0       ; when model run stopped in hours
nfiles    =  96         ; number of output files
date      = '20090705'  ; date in yyyymmdd; string variable!

;============================================================================

;----------------------------------------------------------------------------
; Section 1: Obtain data
;----------------------------------------------------------------------------

; Parameters for loop control - adjust for chosen run-control parameters
;----------------------------------------------------------------------------

print
print, 'DATE: ', date
print
nsteps    =  nfiles 
tstride   =  ROUND(hinclight * 3600.0)
tstart    =  tstride
tend      = nsteps * tstride  

; Arrays
;-------

arlen = FLTARR(nfiles)   ; Float (!) 

; Character strings
;------------------

dummy     = ''
dummy_arr = FLTARR(3)
header    = STRARR(78, 3) 

;----------------------------------------------------------------------------
; Section 1.1:  Go through all files
;----------------------------------------------------------------------------

print, tstart, tend, tstride

print, 'Reading input files:', nfiles

FOR j = tstart, tend, tstride DO BEGIN

  print, 'Time increment: ', j

  ; Create file specification string

   IF (j GE 1 AND j LT 10) THEN $
     time_arr = '0000'+STRCOMPRESS(j, /remove_all)
   IF (j GE 10 AND j LT 100) THEN $
     time_arr = '000'+STRCOMPRESS(j, /remove_all)
   IF (j GE 100 AND j LT 1000) THEN $
     time_arr = '00'+STRCOMPRESS(j, /remove_all)
   IF (j GE 1E3 AND j LT 1E4) THEN $
     time_arr = '0'+STRCOMPRESS(j, /remove_all)
   IF (j GE 1E4 AND j LT 1E5) THEN $
     time_arr = STRCOMPRESS(j, /remove_all)

   filename = $
   '/athome/dahl_jo/Programs/idl/lightning/lightning_cosmo/'+ $
    date+'/'+type+'/lightning_' +                             $
    time_arr + '.txt'

  ; Read dummy to count the lines in the file

  Get_Lun, U
  OpenR, U, filename 
    ReadF, U, dummy
    ReadF, U, dummy
    ReadF, U, dummy
    ReadF, U, dummy
    count_lines = 0 
    REPEAT BEGIN
      ReadF, U, dummy
      count_lines = count_lines + 1
    ENDREP UNTIL EOF(U)
  CLOSE, U
  Free_Lun, U

  n = LONG(count_lines)
 
  index =  j / tstride - 1

  arlen[index] = FLOAT(n)

  ; Define arrays

  def_time = 'time_'+time_arr+' = FLTARR(n)'
  exe_time = EXECUTE(def_time)

  def_lon = 'lon_'+time_arr+' = FLTARR(n)'
  exe_lon = EXECUTE(def_lon)

  def_lat = 'lat_'+time_arr+' = FLTARR(n)'
  exe_lat = EXECUTE(def_lat)

  ; Some exception handling

  IF (exe_time EQ 0 OR exe_lon EQ 0 OR exe_lat EQ 0) THEN BEGIN
    print, 'CONVERT_COSMO: ERROR'
    print, 'Array allocation failed'
    STOP
  ENDIF

  ; Read the data

  Get_Lun, U
  OpenR, U, filename 
    READF, U, dummy
    READF, U, dummy
    READF, U, dummy
    READF, U, dummy
    FOR i = 0, n-1 DO BEGIN
      ReadF, U, dummy_arr 
    
      def_read_time = 'time_'+time_arr+'(i) = dummy_arr[0]'
      def_read_lon  = 'lon_' +time_arr+'(i) = dummy_arr[1]'
      def_read_lat  = 'lat_' +time_arr+'(i) = dummy_arr[2]'
      
      exe_read_time = EXECUTE(def_read_time)
      exe_read_lon  = EXECUTE(def_read_lon)
      exe_read_lat  = EXECUTE(def_read_lat)

      IF (exe_read_time EQ 0 OR exe_read_lon EQ 0 OR exe_read_lat EQ 0) $
        THEN BEGIN
        print, 'CONVERT_COSMO: ERROR'
        print, 'Reading files failed'
        STOP
      ENDIF	
    ENDFOR
  CLOSE, U
  Free_Lun, U

ENDFOR        ; go through files

;----------------------------------------------------------------------------
; Section 2:  Concatenate input arrays into three long arrays
;----------------------------------------------------------------------------

print, 'Concatenating input arrays ...'

arlen_tot = 0

FOR k = 0, n_elements(arlen)-1 DO BEGIN  
  arlen_tot = arlen_tot + arlen(k)
ENDFOR  

; Initialize concatenated arrays

time_tot = FLTARR(arlen_tot)
lon_tot  = FLTARR(arlen_tot)
lat_tot  = FLTARR(arlen_tot)

; add a zero

arlen = [0, arlen]

istart = 0
iend   = 0

FOR j = tstart, tend, tstride DO BEGIN 

print, 'time increments: ', j

  ; Recreate file specification string

  IF (j GE 1 AND j LT 10) THEN $
    time_arr = '0000'+STRCOMPRESS(j, /remove_all)
  IF (j GE 10 AND j LT 100) THEN $ 
    time_arr = '000'+STRCOMPRESS(j, /remove_all)
  IF (j GE 100 AND j LT 1000) THEN $
    time_arr = '00'+STRCOMPRESS(j, /remove_all)
  IF (j GE 1E3 AND j LT 1E4) THEN $
    time_arr = '0'+STRCOMPRESS(j, /remove_all)
  IF (j GE 1E4 AND j LT 1E5) THEN $
    time_arr = STRCOMPRESS(j, /remove_all)

  index = j / tstride - 1

  istart = istart + arlen[index]
  iend   = istart  + arlen[index+1]-1

print, 'istart, iend', j, istart, iend

  def_ti = 'time_tot[istart:iend] = time_' + time_arr
  def_lo = 'lon_tot[istart:iend] = lon_' + time_arr
  def_la = 'lat_tot[istart:iend] = lat_' + time_arr

  edef_ti = EXECUTE(def_ti)
  edef_lo = EXECUTE(def_lo)
  edef_la = EXECUTE(def_la)

  IF (edef_ti EQ 0 OR edef_lo EQ 0 OR edef_la EQ 0) THEN BEGIN
    print, 'Concatenation failed (execute statement) - aborting.'
    STOP
  ENDIF

ENDFOR

;----------------------------------------------------------------------------
; Section 2.1: Remove -999 dummy values, but let array names unchanged;
;              First check whether some lightning occurred at all!
;----------------------------------------------------------------------------

; Abort program if no lightning was simulated in the period

no_lightning = WHERE(FIX(lon_tot) NE -999, no_li_amount)

IF no_li_amount EQ 0 THEN BEGIN
  print, '-----------------------------------------------------------------'
  print, '% CONVERT_COSMO WARNING: No lightning occurred during the period!'
  print, '              ***  Aborting now  ***                             '
  print, '-----------------------------------------------------------------'
  STOP 
ENDIF

rid = WHERE(lon_tot GE -200 AND lat_tot GT -200 , rid_amount)

llon_tot  = lon_tot [rid]
llat_tot  = lat_tot [rid]
ltime_tot = time_tot[rid]

lon_tot   = llon_tot
lat_tot   = llat_tot
time_tot  = ltime_tot

;----------------------------------------------------------------------------
; Section 2.2: order arrays wrt to steadily increasing time
;----------------------------------------------------------------------------

sorted = SORT(ltime_tot)

ltime_tot_srt = time_tot[sorted]
lon_tot_srt   = lon_tot[sorted]
lat_tot_srt   = lat_tot[sorted]

lon_tot  = lon_tot_srt
lat_tot  = lat_tot_srt
time_tot = ltime_tot_srt

;----------------------------------------------------------------------------
; Section 3: Convert locations (rotated coordinates) to 
;            geographical coordinates
;----------------------------------------------------------------------------

 ; COSMO-DE constants

print, 'Converting coordinates ...'

start_lat = -5.0
start_lon = -5.0
dlat      = 0.025
dlon      = 0.025
deg2rad   = !pi/180

pollat    = 40.0
pollon    = -170.0

ie  = 421
je  = 461
ke  = 50
kem = ke-1

bowi = 180./!pi
wibo = !pi/180.
lat0 = 40.0 * wibo
lon0 = -170.0 * wibo
R    = 6371.229

cen_lat = lat_tot * wibo
cen_lon = lon_tot * wibo

; Latitude:
;----------
term1  = cos(lat0) * cos(cen_lat) * cos(cen_lon)
term2  = sin(lat0) * sin(cen_lat)
gcen_lat = asin(term1 + term2)

; Longitude:
;-----------
zah    = cos(cen_lat) * sin(cen_lon)
term3  = sin(cen_lat) * cos(lat0)
term4  = cos(cen_lat) * sin(lat0) * cos(cen_lon)
nen    = term3 - term4
nen    = (nen EQ 0.0) * 1.E-10 + (nen NE 0.0) * nen
gcen_lon = lon0 - atan(zah,nen)

; In degrees, rather than radiants:
;----------------------------------
gcen_lat  = gcen_lat * bowi
gcen_lon  = gcen_lon * bowi
gcen_lon = (gcen_lon LT -180. OR gcen_lon GT 180.) * (gcen_lon + 360) + $
           (gcen_lon LE 180. AND gcen_lon GE -180.) * gcen_lon

lon = gcen_lon
lat = gcen_lat

;----------------------------------------------------------------------------
; Section 4: Crop data to include only LINET domain
;        LINET boundaries: [47.0, 5.0, 51.0, 14.0]
;----------------------------------------------------------------------------

IF linet EQ 1 THEN BEGIN

  lon_ut  = lon
  lat_ut  = lat
  time_ut = time_tot

  crop = WHERE(lon_ut GE 5.0 AND lon_ut LE 14.0 AND $
               lat_ut GE 47.0 AND lat_ut LE 51.0, amount)

  print, amount, ' flashes found in the LINET domain'

  IF amount EQ 0 THEN BEGIN
    print, 'No lightning occurred in the LINET domain during the selected interval'
    GOTO, exception
  ENDIF

  lon_cropped = lon_ut[crop]
  lon     = lon_cropped

  lat_cropped = lat_ut[crop]
  lat     = lat_cropped

  time_cropped = time_ut[crop]
  time_tot     = time_cropped

ENDIF

;----------------------------------------------------------------------------
; Section 5: Convert time from seconds to hh:hh:ss format
;  Brute-force method is used, this application is not extremely 
;  time-critical, so looping through the array and checking every element
;  separately.
;----------------------------------------------------------------------------

nt      = N_ELEMENTS(time_tot)
hours   = FLTARR(nt)
minutes = FLTARR(nt)
seconds = FLTARR(nt)
dummy_t = FLTARR(nt)

FOR i = 0, nt-1 DO BEGIN

  t_util = time_tot(i) / 3600.

  hh  = t_util
  hhr = ROUND(hh * 10.0E4)
  hhr = hhr / 10.0E4
    
  mm = (t_util - FIX(hhr)) * 60.
  mmr = ROUND(mm * 10.0E4)
  mmr = mmr / 10.0E4
    
  ss = (mm - FIX(mmr)) * 60.
  ssr = ROUND(ss * 10.0E4)
  ss = ssr / 10.0E4
    
  hours(i)   = FIX(hh)
  minutes(i) = FIX(mm)
  seconds(i) = FIX(ss)

ENDFOR

;----------------------------------------------------------------------------
; Section 6: Write data in LINET-compatible format
;----------------------------------------------------------------------------

filepath       = ''
filename_final = filepath + '/cosmo_'+date+'.dat'

print, 'Writing data ...', filename_final

datec = (STRMID(date, 2, 6))
datec = LONG(datec)

;----------------------------------------------------------------------------
; Output example (header not included in output file):
;
; date   h  m s   2x dummy lat       lon   
;
; 80822  0  1 14   0   0   47.18475  11.66800
; 80822  0  1 31   0   0   47.51185  13.35095
; 80822  0  1 33   0   0   47.52992  13.37988
;
; date format is ymmdd.
; h, m, s are hours, minutes, and seconds after initialization time, 
; respectively.  lat and lon are latitude and longitude.
; The "dummy" zeros are micro- and miliseconds, which are not determined by 
; the COSMO-scheme.
;---------------------------------------------------------------------------- 

GET_LUN, U
  OpenW, U, filename_final
  FOR i = 0, nt-1 DO BEGIN
    PrintF, U, format='(i6, 3i3, 2i4, f11.5, f10.5)', $
    datec, hours(i), minutes(i), seconds(i), dummy_t(i), dummy_t[i], lat(i), lon(i)
  ENDFOR
  CLOSE, U
FREE_LUN, U

exception:

;============================================================================
; End of program
;============================================================================

END
